package ru.yandex.logbroker2;

import java.io.Closeable;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.http.util.server.UpstreamStat;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerLockMessage;

public class LBTopicContext implements Closeable {
    private static final long ACTIVITY_TIMEOUT = 60000;

    private final Deque<LBMessage> messages = new ArrayDeque<>();
    private final AtomicBoolean consuming = new AtomicBoolean(false);
    private final AtomicLong dataSize = new AtomicLong(0);
    private final AtomicInteger count = new AtomicInteger(0);
    private final LongAdder messagesProcessedCount = new LongAdder();
    private final LongAdder requiredFieldMissingCount = new LongAdder();
    private final LongAdder mandatoryFieldMissingCount = new LongAdder();
    private final LongAdder parseFailedCount = new LongAdder();
    private final String key;
    private final String topic;
    private final String host; // there can be only one host for each topic
    private final Consumer<? super UpstreamStat> stater;
    private final int partition;
    private final int sendBatchSize;
    private final Logger logger;
    private final long readOffset;
    private final long generation;
    private volatile boolean closed = false;
    private long lastSendLagUpdate;
    private long sendLag;
    private long lastReceiveLagUpdate;
    private long receiveLag;

    public LBTopicContext(
        final String key,
        final String host,
        final ConsumerLockMessage lock,
        final Consumer<? super UpstreamStat> stater,
        final int sendBatchSize,
        final Logger logger)
    {
        this.key = key;
        this.host = host;
        this.stater = stater;
        this.sendBatchSize = sendBatchSize;
        this.logger = logger;
        topic = lock.getTopic();
        partition = lock.getPartition();
        readOffset = lock.getReadOffset();
        generation = lock.getGeneration();
    }

    @Override
    public void close() {
        closed = true;
    }

    public boolean startConsume() {
        return consuming.compareAndSet(false, true);
    }

    public void stopConsume() {
        consuming.set(false);
    }

    public String key() {
        return key;
    }

    public String topic() {
        return topic;
    }

    public int partition() {
        return partition;
    }

    public int sendBatchSize() {
        return sendBatchSize;
    }

    public long generation() {
        return generation;
    }

    public long readOffset() {
        return readOffset;
    }

    public boolean closed() {
        return closed;
    }

    public LBMessage poll() {
        return messages.poll();
    }

    public void offer(final LBMessage message) {
        messages.offer(message);
        dataSize.addAndGet(message.data().length);
        count.incrementAndGet();
    }

    public void decreaseDataSize(final long size, final int count) {
        dataSize.addAndGet(-size);
        this.count.addAndGet(-count);
    }

    public long dataSize() {
        return dataSize.get();
    }

    public int count() {
        return count.get();
    }

    public Deque<LBMessage> messages() {
        return messages;
    }

    public Consumer<? super UpstreamStat> stater() {
        return stater;
    }

    public void updateSendLag(final long lag) {
        sendLag = lag;
        lastSendLagUpdate = System.currentTimeMillis();
    }

    public void updateReceiveLag(final long lag) {
        receiveLag = lag;
        lastReceiveLagUpdate = System.currentTimeMillis();
    }

    public boolean active() {
        long now = System.currentTimeMillis();
        return
            (now - lastReceiveLagUpdate < ACTIVITY_TIMEOUT)
            || (now - lastSendLagUpdate < ACTIVITY_TIMEOUT);
    }

    public long sendLag() {
        long diff = System.currentTimeMillis() - lastSendLagUpdate;
        float k = diff / (float) ACTIVITY_TIMEOUT;
        long lag = sendLag - (long) (k * (float) sendLag);
        return Math.max(0, lag);
    }

    public long receiveLag() {
        long diff = System.currentTimeMillis() - lastReceiveLagUpdate;
        float k = diff / (float) ACTIVITY_TIMEOUT;
        long lag = receiveLag - (long) (k * (float) receiveLag);
        return Math.max(0, lag);
    }

    public String host() {
        return host;
    }

    public void messagesProcessed(final long count) {
        messagesProcessedCount.add(count);
    }

    public long messagesProcessedCount() {
        return messagesProcessedCount.sum();
    }

    public void requiredFieldMissing(
        final String fieldName,
        final LBMessage message)
    {
        logger.info(
            "Required field " + fieldName + " value is missing for message "
            + message);
        requiredFieldMissingCount.increment();
    }

    public long requiredFieldMissingCount() {
        return requiredFieldMissingCount.sum();
    }

    public void mandatoryFieldMissing(
        final String fieldName,
        final LBMessage message)
    {
        logger.info(
            "Mandatory field " + fieldName + " value is missing for message "
            + message);
        mandatoryFieldMissingCount.increment();
    }

    public long mandatoryFieldMissingCount() {
        return mandatoryFieldMissingCount.sum();
    }

    public void parseFailed(final Exception e, final LBMessage message) {
        logger.log(Level.WARNING, "Failed to process message " + message, e);
        parseFailedCount.increment();
    }

    public long parseFailedCount() {
        return parseFailedCount.sum();
    }
}
