package ru.yandex.logbroker2;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiFunction;
import java.util.function.Supplier;
import java.util.logging.Level;

import io.grpc.internal.DnsNameResolverProvider;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.NettyChannelBuilder;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;

import ru.yandex.client.tvm2.Tvm2TicketSupplier;
import ru.yandex.function.ConstSupplier;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.StringBuilderable;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.kikimr.persqueue.LogbrokerClientAsyncFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.StreamConsumer;
import ru.yandex.kikimr.persqueue.consumer.StreamListener;
import ru.yandex.kikimr.persqueue.consumer.StreamListener.ReadResponder;
import ru.yandex.kikimr.persqueue.consumer.stream.StreamConsumerConfig;
import ru.yandex.kikimr.persqueue.consumer.transport.message.CommitMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerInitResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerLockMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReleaseMessage;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.kikimr.persqueue.ydb.YdbCoreConfig;
import ru.yandex.logbroker2.config.ImmutableIamJwtConfig;
import ru.yandex.logbroker2.config.ImmutableLogbroker2SingleConsumerConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.GolovanAlertsConfig;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class Logbroker2SingleConsumer
    implements GenericAutoCloseable<IOException>, PrefetchNotifier, Stater
{
    private static final int CONSUMER_RESTART_DELAY = 10000;
    private static final int PREFETCH_FULL_SLEEP = 50;

    private final ConcurrentHashMap<String, LBTopicContext> topicContexts =
        new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, UpstreamStater> upstreamStaters =
        new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, StreamConsumer> consumers =
        new ConcurrentHashMap<>();
    private final AtomicLong prefetchedDataSize = new AtomicLong(0);
    private final AtomicInteger prefetchedCount = new AtomicInteger(0);
    private final Lock prefetchLock = new ReentrantLock();
    private final Condition prefetchAllowed = prefetchLock.newCondition();
    private final HttpProxy<?> server;
    private final ImmutableLogbroker2SingleConsumerConfig consumerConfig;
    private final PrefixedLogger logger;
    private final String statsPrefix;
    private final MessageSender messageSender;
    private final long prefetchAllowLowDataThreshold;
    private final long prefetchAllowLowCount;
    private final long perTopicPrefetchAllowLowDataThreshold;
    private final long perTopicPrefetchAllowLowCount;
    private final Supplier<Credentials> credentialsProvider;

    public Logbroker2SingleConsumer(
        final HttpProxy<?> server,
        final ImmutableLogbroker2SingleConsumerConfig consumerConfig,
        final PrefixedLogger logger,
        final String name,
        final Tvm2TicketSupplier tvm2TicketSupplier)
        throws ConfigException, IOException
    {
        this.server = server;
        this.consumerConfig = consumerConfig;
        this.logger = logger;
        if ("default".equals(name)) {
            this.statsPrefix = "";
        } else {
            this.statsPrefix = name + '-';
        }
        ImmutableURIConfig targetConfig = consumerConfig.targetConfig();
        messageSender = new MessageSender(
            server.client("Target-" + name, targetConfig)
                .adjustAsteriskStater(),
            consumerConfig,
            this,
            logger);

        prefetchAllowLowDataThreshold =
            consumerConfig.prefetchDataSize() >> 1;
        prefetchAllowLowCount = consumerConfig.prefetchCount() >> 1;
        perTopicPrefetchAllowLowDataThreshold =
            consumerConfig.perTopicPrefetchDataSize() >> 1;
        perTopicPrefetchAllowLowCount =
            consumerConfig.perTopicPrefetchCount() >> 1;

        final String tvmClientId = consumerConfig.tvmClientId();
        if (tvmClientId == null) {
            ImmutableIamJwtConfig iamConfig = consumerConfig.iamConfig();
            if (iamConfig == null) {
                credentialsProvider = new ConstSupplier<>(Credentials.none());
            } else {
                credentialsProvider =
                    new IamJwtCredentialsProvider(
                        logger.addPrefix("iam"),
                        iamConfig);
            }
        } else {
            credentialsProvider =
                () -> Credentials.tvm(tvm2TicketSupplier.ticket(tvmClientId));
        }
    }

    public void start() throws IOException {
        for (String balancerHost: consumerConfig.balancerHosts()) {
            startConsumer(balancerHost);
        }
    }

    @Override
    public void close() {
        RuntimeException error = null;
        for (String key: consumers.keySet()) {
            try {
                consumers.get(key).stopConsume();
            } catch (RuntimeException e) {
                error = e;
                logger.log(
                    Level.SEVERE,
                    "StreamConsumer close error, balancer: " + key,
                    e);
            }
        }
        consumers.clear();
        if (error != null) {
            throw error;
        }
    }

    private void startConsumer(final String balancerHost) throws IOException {
        final StreamConsumerConfig consumerConfig =
            StreamConsumerConfig.builder(
                this.consumerConfig.topics(),
                this.consumerConfig.clientId())
                    .configureSession((x) -> {
                        x.setClientSideLocksAllowed(true);
                        x.setReadOnlyLocal(
                            this.consumerConfig.readOnlyLocal());
                    })
                    .setCredentialsProvider(credentialsProvider)
                    .build();
        LogbrokerClientAsyncFactory factory;
        if (this.consumerConfig.iamConfig() == null) {
            ProxyBalancer balancer = new ProxyBalancer(
                balancerHost,
                this.consumerConfig.balancerPort(),
                this.consumerConfig.proxyPort());
            factory = new LogbrokerClientAsyncFactory(balancer);
        } else {
            ImmutableIamJwtConfig iamConfig = this.consumerConfig.iamConfig();
            logger.info(
                "Starting consumer for " + this.consumerConfig.clientId()
                + " iam " + balancerHost + " endpoint "
                + iamConfig.discoveryEndpoint()
                + " " + iamConfig.database());
            NettyChannelBuilder channelBuilder =
                NettyChannelBuilder
                    .forAddress(
                        balancerHost,
                        this.consumerConfig.balancerPort())
                    .nameResolverFactory(new DnsNameResolverProvider());


            if (iamConfig.https().trustManagerFactory() != null) {
                channelBuilder.sslContext(
                    GrpcSslContexts.forClient()
                        .trustManager(InsecureTrustManagerFactory.INSTANCE)
                        //.trustManager(iamConfig.https().trustManagerFactory())
                        .build());
            } else {
                channelBuilder.usePlaintext();
            }

            factory = new LogbrokerClientAsyncFactory(
                new YdbCoreConfig(
                    //""
                    iamConfig.discoveryEndpoint(),
                    //""
                    iamConfig.database(),
                    true));
        }

        factory.streamConsumer(consumerConfig).handle(new ConsumerStartCallback(balancerHost));
    }

    private void startConsume(final String balancerHost) {
        consumers.get(balancerHost).startConsume(
            new StreamConsumerListener(balancerHost));
    }

    private void sendMessages(
        final List<MessageBatch> messages,
        final ReadResponder readResponder)
    {
        Iterator<MessageBatch> batchIter = messages.iterator();
        long currentTime = System.currentTimeMillis();
        while (batchIter.hasNext()) {
            final MessageBatch batch = batchIter.next();
            final boolean lastBatch;
            if (!batchIter.hasNext()) {
                //last batch entry
                lastBatch = true;
            } else {
                lastBatch = false;
            }
            final String topicKey =
                Integer.toString(batch.getPartition()) + '@' + batch.getTopic();
            final LBTopicContext topicContext = topicContexts.get(topicKey);
            if (topicContext == null) {
                logger.severe("No topic context exists for topicKey: "
                    + topicKey + ". Skipping batch");
            } else {
                Iterator<MessageData> dataIter =
                    batch.getMessageData().iterator();
                while (dataIter.hasNext()) {
                    final MessageData data = dataIter.next();
                    final ReadResponder setResponder;
                    if (lastBatch && !dataIter.hasNext()) {
                        setResponder = readResponder;
                    } else {
                        setResponder = null;
                    }
                    LBMessage lbMessage = new LBMessage(data, setResponder);
                    final long lag = currentTime - lbMessage.writeTime();
                    if (logger.isLoggable(Level.FINE)) {
                        StringBuilder log =
                            new StringBuilder("Received message: topic=");
                        log.append(topicKey);
                        log.append(", message=");
                        lbMessage.toStringBuilder(log);
                        log.append(", lag=");
                        log.append(lag);
                        if (server.debugFlags().contains("log-payload")) {
                            log.append(", payload = <");
                            log.append(
                                new String(
                                    lbMessage.data(),
                                    StandardCharsets.UTF_8));
                            log.append('>');
                        }
                        logger.fine(new String(log));
                    }
                    topicContext.updateReceiveLag(lag);
                    synchronized (topicContext) {
                        topicContext.offer(lbMessage);
                    }
                    long totalDataSize =
                        prefetchedDataSize.addAndGet(lbMessage.data().length);
                    int totalMessages =
                        prefetchedCount.incrementAndGet();
                    while ((totalDataSize > consumerConfig.prefetchDataSize()
                        || totalMessages > consumerConfig.prefetchCount()
                        || topicContext.dataSize()
                            > consumerConfig.perTopicPrefetchDataSize()
                        || topicContext.count()
                            > consumerConfig.perTopicPrefetchCount())
                        && !topicContext.closed())
                    {
                        notifyConsume(topicContext);
                        prefetchLock.lock();
                        try {
                            if (logger.isLoggable(Level.FINE)) {
                                StringBuilder log =
                                    new StringBuilder(
                                        "Prefetch overrun: size=");
                                log.append(totalDataSize);
                                log.append(", count=");
                                log.append(totalMessages);
                                log.append(", topicSize=");
                                log.append(topicContext.dataSize());
                                log.append(", topicCount=");
                                StringBuilderable.toStringBuilder(
                                    log,
                                    topicContext.messages());
                                logger.fine(new String(log));
                            } else {
                                logger.info(
                                    "Prefetch overrun: size=" + totalDataSize
                                        + ", count=" + totalMessages
                                        + ", topicSize="
                                        + topicContext.dataSize()
                                        + ", topicCount="
                                        + topicContext.messages().size());
                            }

                            prefetchAllowed.await(
                                PREFETCH_FULL_SLEEP,
                                TimeUnit.MILLISECONDS);
                        } catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        } finally {
                            prefetchLock.unlock();
                        }
                        totalDataSize = prefetchedDataSize.get();
                        totalMessages = prefetchedCount.get();
                    }
                }
                notifyConsume(topicContext);
            }
        }
    }

    private void notifyConsume(final LBTopicContext topicContext) {
        if (topicContext.closed()) {
            return;
        }
        if (topicContext.startConsume()) {
            messageSender.sendMessages(topicContext);
        }
    }

    @Override
    public void notifyPrefetch(
        final List<LBMessage> messages,
        final LBTopicContext topicContext)
    {
        long dataSize = 0;
        int count = messages.size();
        for (int i = 0; i < count; ++i) {
            dataSize += messages.get(i).data().length;
        }
        final long currentDataSize = prefetchedDataSize.getAndAdd(-dataSize);
        final int currentCount = prefetchedCount.getAndAdd(-count);
        final long topicDataSize = topicContext.dataSize();
        final int topicCount = topicContext.count();
        topicContext.decreaseDataSize(dataSize, count);
        if ((currentDataSize > prefetchAllowLowDataThreshold
                && prefetchedDataSize.get() < prefetchAllowLowDataThreshold)
            || (currentCount > prefetchAllowLowCount
                && prefetchedCount.get() < prefetchAllowLowCount)
            || (topicDataSize > perTopicPrefetchAllowLowDataThreshold
                && topicContext.dataSize() < perTopicPrefetchAllowLowDataThreshold)
            || (topicCount > perTopicPrefetchAllowLowCount
                && topicContext.count() < perTopicPrefetchAllowLowCount))
        {
            prefetchLock.lock();
            try {
                prefetchAllowed.signalAll();
            } finally {
                prefetchLock.unlock();
            }
        }
    }

    private class StreamConsumerListener implements StreamListener {
        private final String balancerHost;

        StreamConsumerListener(final String balanceHost) {
            this.balancerHost = balanceHost;
        }

        @Override
        public void onInit(final ConsumerInitResponse init) {
            logger.info(
                "OnInit: " + init.getSessionId()
                + ", balancer host: " + balancerHost);
        }

        @Override
        public void onRead(
            final ConsumerReadResponse read,
            final ReadResponder readResponder)
        {
            sendMessages(read.getBatches(), readResponder);
        }

        @Override
        public void onCommit(final CommitMessage commit) {
            logger.info("OnCommit: " + commit.getCookies());
        }

        @Override
        public void onLock(
            final ConsumerLockMessage lock,
            final LockResponder lockResponder)
        {
            final String topicKey =
                Integer.toString(lock.getPartition()) + '@' + lock.getTopic();
            logger.info("OnLock: " + topicKey);
            UpstreamStater stater = upstreamStaters.get(lock.getTopic());
            if (stater == null) {
                UpstreamStater newStater =
                    new UpstreamStater(
                        server.config().metricsTimeFrame(),
                        statsPrefix + lock.getTopic());
                stater =
                    upstreamStaters.putIfAbsent(lock.getTopic(), newStater);
                if (stater == null) {
                    stater = newStater;
                    server.registerStater(stater);
                }
            }
            String topic = lock.getTopic();
            Integer sendBatchSize =
                consumerConfig.sendBatchSize().get(topic);
            if (sendBatchSize == null) {
                for (Map.Entry<String, Integer> entry
                    : consumerConfig.sendBatchSize().entrySet())
                {
                    String entryTopic = entry.getKey();
                    if (entryTopic != null && topic.contains(entryTopic)) {
                        sendBatchSize = entry.getValue();
                        break;
                    }
                }
                if (sendBatchSize == null) {
                    sendBatchSize = consumerConfig.sendBatchSize().get(null);
                }
                if (sendBatchSize == null) {
                    sendBatchSize = 1;
                }
            }
            LBTopicContext oldContext =
                topicContexts.put(
                    topicKey,
                    new LBTopicContext(
                        topicKey,
                        balancerHost,
                        lock,
                        stater,
                        sendBatchSize,
                        logger.addPrefix(topicKey)));
            if (oldContext != null) {
                oldContext.close();
            }
            logger.info("OnLock: " + lock.getTopic() + '/'
                + lock.getPartition() + '/'
                + lock.getReadOffset() + '/'
                + lock.getEndOffset() + '/'
                + lock.getGeneration()
                + ", sendBatchSize: " + sendBatchSize);
            lockResponder.locked(lock.getReadOffset(), false);
        }

        @Override
        public void onRelease(final ConsumerReleaseMessage release) {
            final String topicKey =
                Integer.toString(release.getPartition())
                    + '@' + release.getTopic();
            logger.info("OnRelease: " + topicKey);

            final LBTopicContext oldContext = topicContexts.remove(topicKey);
            if (oldContext != null) {
                oldContext.close();
            }
        }

        @Override
        public void onClose() {
            logger.info("OnClose(), balancer host: " + balancerHost);
            closeTopicContexts();
        }

        @Override
        public void onError(final Throwable e) {
            logger.log(
                Level.SEVERE,
                "OnError, balancer host: " + balancerHost,
                e);
            closeTopicContexts();
            try {
                Thread.sleep(CONSUMER_RESTART_DELAY);
            } catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
            consumers.remove(balancerHost);
            try {
                startConsumer(balancerHost);
            } catch (IOException ioe) {
                logger.log(Level.SEVERE, "Failed to restart consumer for " + balancerHost, ioe);
                Runtime.getRuntime().halt(1);
            }
        }

        private void closeTopicContexts() {
            List<String> keys = new ArrayList<>();
            topicContexts.forEach((key, value) -> {
                if (value.host().equals(balancerHost)) {
                    keys.add(key);
                }
            });
            for (String key: keys) {
                LBTopicContext context = topicContexts.remove(key);
                if (context != null) {
                    context.close();
                }
            }
        }
    }

    private class ConsumerStartCallback
        implements BiFunction<StreamConsumer, Throwable, Void>
    {
        private final String balancerHost;

        ConsumerStartCallback(final String balancerHost) {
            this.balancerHost = balancerHost;
        }

        @Override
        public Void apply(final StreamConsumer consumer, final Throwable t) {
            if (consumer == null || t != null) {
                logger.log(Level.SEVERE, "StreamConsumer start error", t);
                close();
                Runtime.getRuntime().halt(1);
            } else {
                Logbroker2SingleConsumer.this.consumers.put(
                    balancerHost,
                    consumer);
                startConsume(balancerHost);
                logger.info("Created StreamConsumer: " + consumer);
            }
            return null;
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        statsConsumer.stat(
            statsPrefix + "prefetch-buffer-size_ammm",
            prefetchedDataSize.get());
        statsConsumer.stat(
            statsPrefix + "prefetch-count_ammm",
            prefetchedCount.get());
        final boolean prefetchSaturated =
            (prefetchedDataSize.get() > prefetchAllowLowDataThreshold)
                || (prefetchedCount.get() > prefetchAllowLowCount);
        statsConsumer.stat(
            statsPrefix + "prefetch-saturated_ammm",
            prefetchSaturated);

        HashMap<String, List<LBTopicContext>> topicContexts =
            new HashMap<>();
        for (LBTopicContext context: this.topicContexts.values()) {
            String topic = context.topic();
            List<LBTopicContext> contexts = topicContexts.get(topic);
            if (contexts == null) {
                contexts = new ArrayList<>(2 + 2 * 2);
                topicContexts.put(topic, contexts);
            }
            contexts.add(context);
        }

        int totalLockedPartitions = 0;
        int totalActivePartitions = 0;
        long totalMaxSendLag = 0L;
        long totalMaxReceiveLag = 0L;
        long totalMessagessProcessedCount = 0L;
        long totalRequiredFieldMissingCount = 0L;
        long totalMandatoryFieldMissingCount = 0L;
        long totalParseFailedCount = 0L;
        for (Map.Entry<String, List<LBTopicContext>> entry
            : topicContexts.entrySet())
        {
            final String topic = entry.getKey();
            int maxSendLagPartition = 0;
            long maxSendLag = 0;
            int maxReceiveLagPartition = 0;
            long maxReceiveLag = 0;
            int activePartitions = 0;
            long messagesProcessedCount = 0L;
            long requiredFieldMissingCount = 0L;
            long mandatoryFieldMissingCount = 0L;
            long parseFailedCount = 0L;
            for (LBTopicContext context: entry.getValue()) {
                if (maxSendLag < context.sendLag()) {
                    maxSendLag = context.sendLag();
                    maxSendLagPartition = context.partition();
                }
                if (maxReceiveLag < context.receiveLag()) {
                    maxReceiveLag = context.receiveLag();
                    maxReceiveLagPartition = context.partition();
                }
                if (context.active()) {
                    activePartitions++;
                }
                messagesProcessedCount += context.messagesProcessedCount();
                requiredFieldMissingCount +=
                    context.requiredFieldMissingCount();
                mandatoryFieldMissingCount +=
                    context.mandatoryFieldMissingCount();
                parseFailedCount += context.parseFailedCount();
            }
            int lockedPartitions = entry.getValue().size();
            totalLockedPartitions += lockedPartitions;
            totalActivePartitions += activePartitions;
            if (totalMaxSendLag < maxSendLag) {
                totalMaxSendLag = maxSendLag;
            }
            if (totalMaxReceiveLag < maxReceiveLag) {
                totalMaxReceiveLag = maxReceiveLag;
            }
            totalMessagessProcessedCount += messagesProcessedCount;
            totalRequiredFieldMissingCount += requiredFieldMissingCount;
            totalMandatoryFieldMissingCount += mandatoryFieldMissingCount;
            totalParseFailedCount += totalParseFailedCount;
            statsConsumer.stat(
                StringUtils.concat(statsPrefix, topic, "-max-send-lag_axxx"),
                maxSendLag);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-max-send-lag-partition_ammm"),
                maxSendLagPartition);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-max-receive-lag_axxx"),
                maxReceiveLag);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-max-receive-lag-partition_ammm"),
                maxReceiveLagPartition);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-locked-partitions_ammm"),
                lockedPartitions);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-active-partitions_ammm"),
                activePartitions);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-messages-processed_dmmm"),
                messagesProcessedCount);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-required-field-missing_dmmm"),
                requiredFieldMissingCount);
            statsConsumer.stat(
                StringUtils.concat(
                    statsPrefix,
                    topic,
                    "-mandatory-field-missing_dmmm"),
                mandatoryFieldMissingCount);
            statsConsumer.stat(
                StringUtils.concat(statsPrefix, topic, "-parse-failed_dmmm"),
                parseFailedCount);
        }
        statsConsumer.stat(
            statsPrefix + "locked-partitions_ammm",
            totalLockedPartitions);
        statsConsumer.stat(
            statsPrefix + "active-partitions_ammm",
            totalActivePartitions);
        statsConsumer.stat(statsPrefix + "max-send-lag_axxx", totalMaxSendLag);
        statsConsumer.stat(
            statsPrefix + "max-receive-lag_axxx",
            totalMaxReceiveLag);
        statsConsumer.stat(
            statsPrefix + "messages-processed_dmmm",
            totalMessagessProcessedCount);
        statsConsumer.stat(
            statsPrefix + "required-field-missing_dmmm",
            totalRequiredFieldMissingCount);
        statsConsumer.stat(
            statsPrefix + "mandatory-field-missing_dmmm",
            totalMandatoryFieldMissingCount);
        statsConsumer.stat(
            statsPrefix + "parse-failed_dmmm",
            totalParseFailedCount);
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        String statsPrefix)
    {
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChartGroup health =
            new GolovanChartGroup(statsPrefix, statsPrefix);
        statsPrefix = statsPrefix + this.statsPrefix;

        GolovanChart lockedPartitions = new GolovanChart(
            this.statsPrefix + "locked-partitions",
            " locked partitions",
            false,
            true,
            0d);
        lockedPartitions.addSignal(
            new GolovanSignal(
                "mul(" + statsPrefix + "locked-partitions_ammm,5)",
                config.tag(),
                "locked partitions",
                null,
                0,
                false));
        health.addChart(lockedPartitions);

        GolovanChart activePartitions = new GolovanChart(
            this.statsPrefix + "active-partitions",
            " active partitions",
            false,
            true,
            0d);
        activePartitions.addSignal(
            new GolovanSignal(
                "mul("
                + statsPrefix
                + "active-partitions_ammm,5)",
                config.tag(),
                "active partitions",
                null,
                0,
                false));
        health.addChart(activePartitions);

        GolovanChart sendLag = new GolovanChart(
            this.statsPrefix + "send-lag",
            " max send lag (ms)",
            false,
            false,
            0d);
        sendLag.addSignal(
            new GolovanSignal(
                statsPrefix + "max-send-lag_axxx",
                config.tag(),
                "send lag",
                null,
                0,
                false));
        health.addChart(sendLag);

        GolovanChart receiveLag = new GolovanChart(
            this.statsPrefix + "receive-lag",
            " max receive lag (ms)",
            false,
            false,
            0d);
        receiveLag.addSignal(
            new GolovanSignal(
                statsPrefix + "max-receive-lag_axxx",
                config.tag(),
                "send lag",
                null,
                0,
                false));
        health.addChart(receiveLag);

        panel.addCharts("topics", null, health);

        GolovanChartGroup errors =
            new GolovanChartGroup(statsPrefix, statsPrefix);

        GolovanChart messagesProcessed = new GolovanChart(
            this.statsPrefix + "messages-processed",
            " messages processed (rps)",
            false,
            true,
            0d);
        messagesProcessed.addSignal(
            new GolovanSignal(
                statsPrefix + "messages-processed_dmmm",
                config.tag(),
                "messages processed count",
                null,
                1,
                false));
        errors.addChart(messagesProcessed);

        GolovanChart requiredField = new GolovanChart(
            this.statsPrefix + "required-field-missing",
            " required field missing (rps)",
            false,
            true,
            0d);
        requiredField.addSignal(
            new GolovanSignal(
                statsPrefix + "required-field-missing_dmmm",
                config.tag(),
                "required field missed count",
                null,
                1,
                false));
        errors.addChart(requiredField);

        GolovanChart mandatoryField = new GolovanChart(
            this.statsPrefix + "mandatory-field-missing",
            " mandatory field missing (rps)",
            false,
            true,
            0d);
        mandatoryField.addSignal(
            new GolovanSignal(
                statsPrefix + "mandatory-field-missing_dmmm",
                config.tag(),
                "mandatory field missing count",
                null,
                1,
                false));
        errors.addChart(mandatoryField);

        GolovanChart parseFailed = new GolovanChart(
            this.statsPrefix + "parse-failed",
            " parse failed (rps)",
            false,
            true,
            0d);
        parseFailed.addSignal(
            new GolovanSignal(
                statsPrefix + "parse-failed_dmmm",
                config.tag(),
                "parse failed count",
                null,
                1,
                false));
        errors.addChart(parseFailed);

        panel.addCharts("topics", null, errors);
    }

    @Override
    public void addToAlertsConfig(
        final IniConfig alertsConfig,
        final ImmutableGolovanPanelConfig panelConfig,
        String statsPrefix)
        throws BadRequestException
    {
        ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
        statsPrefix = statsPrefix + this.statsPrefix;
        alerts.createAlert(
            alertsConfig,
            alerts.module()
            + '-'
            + GolovanAlertsConfig.clearAlertName(statsPrefix)
            + "send-lag",
            statsPrefix + "max-send-lag_axxx",
            new AlertThresholds(180000d, null, null));
        alerts.createAlert(
            alertsConfig,
            alerts.module()
            + '-'
            + GolovanAlertsConfig.clearAlertName(statsPrefix)
            + "mandatory-field-missing",
            statsPrefix + "mandatory-field-missing_dmmm",
            new AlertThresholds(0.5d, null, null));
        alerts.createAlert(
            alertsConfig,
            alerts.module()
            + '-'
            + GolovanAlertsConfig.clearAlertName(statsPrefix)
            + "parse-failed",
            statsPrefix + "parse-failed_dmmm",
            new AlertThresholds(0.5d, null, null));
    }
}
