package ru.yandex.logbroker2;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;

import ru.yandex.collection.IntPair;
import ru.yandex.compress.Deflater;
import ru.yandex.compress.GzipOutputStream;
import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.logbroker2.config.ImmutableFieldConfig;
import ru.yandex.logbroker2.config.ImmutableLogbroker2SingleConsumerConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.QueryConstructor;

public class MessageSender {
    private static final IntPair<Void> UNPROCESSABLE_RESULT =
        new IntPair<>(HttpStatus.SC_UNSUPPORTED_MEDIA_TYPE, null);
    private static final String TOPIC = "topic=";
    private static final String PARTITION = "&partition=";
    private static final String OFFSET = "&offset=";
    private static final String SEQNO = "&seqNo=";
    private static final String MESSAGE_CREATE_TIME = "&message-create-time=";
    private static final String MESSAGE_WRITE_TIME = "&message-write-time=";
    private final Timer timer = new Timer(true);
    private final AsyncClient client;
    private final PrefetchNotifier prefetchNotifier;
    private final PrefixedLogger logger;
    private final ImmutableURIConfig targetConfig;
    private final String targetUriPrefix;
    private final boolean gzipRequests;
    private final Map<String, ImmutableFieldConfig> fieldsConfig;
    private final DataProcessor dataProcessor;

    public MessageSender(
        final AsyncClient client,
        final ImmutableLogbroker2SingleConsumerConfig consumerConfig,
        final PrefetchNotifier prefetchNotifier,
        final PrefixedLogger logger)
        throws ConfigException
    {
        this.client = client;
        this.prefetchNotifier = prefetchNotifier;
        this.logger = logger;
        targetConfig = consumerConfig.targetConfig();
        targetUriPrefix =
            targetConfig.request() + targetConfig.firstCgiSeparator();
        gzipRequests = consumerConfig.gzipRequests();

        fieldsConfig = consumerConfig.fieldsConfig();
        if (fieldsConfig.isEmpty()) {
            dataProcessor = null;
        } else {
            dataProcessor = consumerConfig.dataFormat().createDataProcessor(
                fieldsConfig);
        }
    }

    public Future<IntPair<Void>> sendMessages(final LBTopicContext context) {
        List<LBMessage> batch = new ArrayList<>();
        synchronized (context) {
            for (int i = 0; i < context.sendBatchSize(); i++) {
                LBMessage message = context.poll();
                if (message == null) {
                    break;
                } else {
                    batch.add(message);
                }
            }
        }
        if (batch.size() == 0) {
            context.stopConsume();
            return null;
        } else if (batch.size() == 1) {
            return sendMessage(batch.get(0), context);
        } else {
            return sendMultipart(batch, context);
        }
    }

    // Returns adjusted uri and headers to be added to request
    // Returns null if message shouldn't be sent at all
    private ProcessingResult processMessage(
        final LBMessage message,
        final LBTopicContext context,
        final String baseUri)
    {
        if (dataProcessor == null) {
            return new ProcessingResult(baseUri, Collections.emptyList());
        }
        try {
            Map<String, List<String>> result =
                dataProcessor.processData(message.data());
            if (logger.isLoggable(Level.FINE)) {
                logger.fine(message + " processing result: " + result);
            }
            QueryConstructor query = new QueryConstructor(baseUri);
            List<Header> headers = new ArrayList<>(fieldsConfig.size());
            for (Map.Entry<String, ImmutableFieldConfig> entry
                : fieldsConfig.entrySet())
            {
                String fieldName = entry.getKey();
                ImmutableFieldConfig fieldConfig = entry.getValue();
                List<String> values = result.get(fieldName);
                if (values == null) {
                    FieldPresencePolicy presence = fieldConfig.presence();
                    if (presence == FieldPresencePolicy.REQUIRED) {
                        context.requiredFieldMissing(fieldName, message);
                        return null;
                    } else if (presence == FieldPresencePolicy.MANDATORY) {
                        context.mandatoryFieldMissing(fieldName, message);
                        return null;
                    }
                } else {
                    int size = values.size();
                    String headerName = fieldConfig.headerName();
                    if (headerName != null) {
                        for (int i = 0; i < size; ++i) {
                            headers.add(
                                HeaderUtils.createHeader(
                                    headerName,
                                    values.get(i)));
                        }
                    }
                    String cgiParameter = fieldConfig.cgiParameter();
                    if (cgiParameter != null) {
                        for (int i = 0; i < size; ++i) {
                            query.append(cgiParameter, values.get(i));
                        }
                    }
                }
            }
            return new ProcessingResult(query.toString(), headers);
        } catch (BadRequestException | LBParseException e) {
            context.parseFailed(e, message);
            return null;
        }
    }

    private static byte[] gzipCompress(final byte[] data) {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(data.length);
        try (GzipOutputStream gzipOS = new GzipOutputStream(
                bos,
                32 * 1024,
                false,
                Deflater.BEST_SPEED))
        {
            gzipOS.write(data);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return bos.toByteArray();
    }

    private Future<IntPair<Void>> sendMessage(
        final LBMessage message,
        final LBTopicContext context)
    {
        String uri =
            targetUriPrefix
            + TOPIC + context.topic()
            + PARTITION + context.partition()
            + OFFSET + message.offset()
            + SEQNO + message.seqNo()
            + MESSAGE_CREATE_TIME + message.createTime()
            + MESSAGE_WRITE_TIME + message.writeTime();
        ProcessingResult processingResult =
            processMessage(
                message,
                context,
                uri);
        MessagesSendCallback callback = new MessagesSendCallback(
            Collections.singletonList(message),
            context);
        if (processingResult == null) {
            callback.completed(UNPROCESSABLE_RESULT);
            return new CompletedFuture<>(UNPROCESSABLE_RESULT);
        }
        uri = processingResult.uri;
        final BasicAsyncRequestProducerGenerator post;
        if (gzipRequests) {
            byte[] gzipped = gzipCompress(message.data());
            post =
                new BasicAsyncRequestProducerGenerator(
                    uri,
                    gzipped,
                    ContentType.DEFAULT_BINARY);
            post.addHeader("Content-Encoding", "gzip");
        } else {
            post =
                new BasicAsyncRequestProducerGenerator(
                    uri,
                    message.data(),
                    ContentType.DEFAULT_BINARY);
        }
        int headersSize = processingResult.headers.size();
        for (int i = 0; i < headersSize; ++i) {
            post.addHeader(processingResult.headers.get(i));
        }
        return client.execute(
            targetConfig.host(),
            post,
            StatusCodeAsyncConsumerFactory.ANY_GOOD,
            new UpstreamStaterFutureCallback<>(callback, context.stater()));
    }

    private Future<IntPair<Void>> sendMultipart(
        final List<LBMessage> messages,
        final LBTopicContext context)
    {
        LBMessage first = messages.get(0);
        LBMessage last = messages.get(messages.size() - 1);
        StringBuilder uri = new StringBuilder(
            targetUriPrefix
            + TOPIC + context.topic()
            + PARTITION + context.partition());
        int uriPrefixLen = uri.length();
        uri.append("&offsets=");
        uri.append(first.offset());
        uri.append('-');
        uri.append(last.offset());
        uri.append("&seqNums=");
        uri.append(first.seqNo());
        uri.append('-');
        uri.append(last.seqNo());
        uri.append("&message-create-times=");
        uri.append(first.createTime());
        uri.append('-');
        uri.append(last.createTime());
        uri.append("&message-write-times=");
        uri.append(first.writeTime());
        uri.append('-');
        uri.append(last.writeTime());
        uri.append("&batch-size=");
        uri.append(messages.size());

        final String commonUri = new String(uri);

        boolean empty = true;
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");
        for (LBMessage message: messages) {
            uri.setLength(uriPrefixLen);
            uri.append(OFFSET);
            uri.append(message.offset());
            uri.append(SEQNO);
            uri.append(message.seqNo());
            uri.append(MESSAGE_CREATE_TIME);
            uri.append(message.createTime());
            uri.append(MESSAGE_WRITE_TIME);
            uri.append(message.writeTime());

            ProcessingResult processingResult = processMessage(
                message,
                context,
                uri.toString());
            if (processingResult != null) {
                empty = false;
                FormBodyPartBuilder partBuilder =
                    FormBodyPartBuilder
                        .create()
                        .addField(YandexHeaders.URI, processingResult.uri)
                        .setBody(
                            new ByteArrayBody(
                                message.data(),
                                ContentType.DEFAULT_BINARY,
                                null))
                        .setName("message.bin");
                int headersSize = processingResult.headers.size();
                for (int i = 0; i < headersSize; ++i) {
                    Header header = processingResult.headers.get(i);
                    partBuilder.addField(
                        header.getName(),
                        header.getValue());
                }

                builder.addPart(partBuilder.build());
            }
        }
        MessagesSendCallback callback =
            new MessagesSendCallback(messages, context);
        if (empty) {
            callback.completed(UNPROCESSABLE_RESULT);
            return new CompletedFuture<>(UNPROCESSABLE_RESULT);
        }
        try {
            final BasicAsyncRequestProducerGenerator post;
            if (gzipRequests) {
                HttpEntity multiPart =
                    builder.build();
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                multiPart.writeTo(baos);
                byte[] data = baos.toByteArray();
                byte[] gzipped = gzipCompress(data);
                post =
                    new BasicAsyncRequestProducerGenerator(
                        commonUri,
                        gzipped,
                        ContentType.get(multiPart));
                post.addHeader("Content-Encoding", "gzip");
            } else {
                post =
                    new BasicAsyncRequestProducerGenerator(
                        commonUri,
                        builder.build());
            }
            return client.execute(
                targetConfig.host(),
                post,
                StatusCodeAsyncConsumerFactory.ANY_GOOD,
                new UpstreamStaterFutureCallback<>(
                    callback,
                    context.stater()));
        } catch (IOException e) {
            //This should never be happen
            logger.log(
                Level.SEVERE,
                "Message batch send failed",
                e);
            scheduleRetry(messages, context);
            return null;
        }
    }

    private void scheduleRetry(
        final List<LBMessage> messages,
        final LBTopicContext context)
    {
        timer.schedule(
            new RetryDelayer(messages, context),
            targetConfig.httpRetries().interval());
    }

    private static class ProcessingResult {
        private final String uri;
        private final List<Header> headers;

        private ProcessingResult(
            final String uri,
            final List<Header> headers)
        {
            this.uri = uri;
            this.headers = headers;
        }
    }

    private class MessagesSendCallback
        implements FutureCallback<IntPair<Void>>
    {
        private final List<LBMessage> messages;
        private final LBTopicContext context;

        MessagesSendCallback(
            final List<LBMessage> messages,
            final LBTopicContext context)
        {
            this.messages = messages;
            this.context = context;
        }

        @Override
        public void completed(final IntPair<Void> code) {
            context.messagesProcessed(messages.size());
            prefetchNotifier.notifyPrefetch(messages, context);
            if (messages.size() == 1) {
                messageSent(messages.get(0), code.first());
            } else {
                logger.fine("Message batch sent: size=" + messages.size());
                for (LBMessage message: messages) {
                    messageSent(message, code.first());
                }
            }
            sendMessages(context);
        }

        private void messageSent(final LBMessage message, final int code) {
            long lag = System.currentTimeMillis() - message.writeTime();
            logger.fine("Message sent: topic=" + context.key()
                + ", message=" + message
                + ", lag=" + lag
                + ", code=" + code);
            context.updateSendLag(lag);
            if (message.readResponder() != null) {
                logger.info("Commiting");
                message.readResponder().commit();
            }
        }

        private String failMessage(final LBMessage message, final int code) {
            return "Message send failed: topic=" + context.key()
                + ",  message=" + message
                + " with code=" + code;
        }

        @Override
        public void failed(final Exception e) {
            final int code;
            if (e instanceof BadResponseException) {
                code = ((BadResponseException) e).statusCode();
            } else {
                code = 0;
            }
            if (messages.size() == 1) {
                logger.log(
                    Level.SEVERE,
                    failMessage(messages.get(0), code),
                    e);
            } else {
                logger.log(
                    Level.SEVERE,
                    "Message batch send failed: topic=" + context.key()
                        + ", offsets="
                        + Long.toString(messages.get(0).offset())
                        + '-'
                        + Long.toString(
                            messages.get(messages.size() - 1).offset()),
                    e);
                for (LBMessage message: messages) {
                    logger.severe(failMessage(message, code));
                }
            }
            scheduleRetry(messages, context);
        }

        @Override
        public void cancelled() {
        }
    }

    private class RetryDelayer extends TimerTask {
        private final List<LBMessage> messages;
        private final LBTopicContext context;

        RetryDelayer(
            final List<LBMessage> messages,
            final LBTopicContext context)
        {
            this.messages = messages;
            this.context = context;
        }

        @Override
        public void run() {
            if (messages.size() == 1) {
                sendMessage(messages.get(0), context);
            } else {
                sendMultipart(messages, context);
            }
        }
    }
}

