package ru.yandex.logbroker2.config;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractIamJwtConfigBuilder
    <T extends AbstractIamJwtConfigBuilder<T>>
     extends AbstractGrpcHostConfigBuilder<T>
     implements IamJwtConfig
{
    private PrivateKey privateKey;
    //private String publicKey;
    private String serviceAccountId;
    private String keyId;
    private String discoveryEndpoint;
    private String database;
    private String audience;

    protected AbstractIamJwtConfigBuilder(final IamJwtConfig config) {
        super(config);
        privateKey(config.privateKey());
        serviceAccountId(config.serviceAccountId());
        keyId(config.keyId());
        database(config.database());
        discoveryEndpoint(config.discoveryEndpoint());
        audience(config.audience());
    }

    protected AbstractIamJwtConfigBuilder(
        final IniConfig config,
        final IamJwtConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        File jsonConfigFile = config.getInputFile("ca_config", null);
        if (jsonConfigFile != null) {
            try {
                JsonObject caObj =
                    TypesafeValueContentHandler.parse(
                        new FileReader(jsonConfigFile, StandardCharsets.UTF_8));
                JsonMap caMap = caObj.asMap();
                String privateKeyStr = caMap.getString("private_key");
                this.serviceAccountId = caMap.getString("service_account_id");
                //this.publicKey = caMap.getString("public_key");
                this.keyId = caMap.getString("id");
                KeyFactory keyFactory = KeyFactory.getInstance("RSA");
                privateKeyStr = privateKeyStr
                    .replace("-----BEGIN PRIVATE KEY-----", "")
                    .replace("-----END PRIVATE KEY-----", "")
                    .replace("\n", "")
                    .trim();
                this.privateKey = keyFactory.generatePrivate(
                    new PKCS8EncodedKeySpec(
                        Base64.getDecoder().decode(privateKeyStr)));
            } catch (IOException | JsonException | NoSuchAlgorithmException | InvalidKeySpecException e) {
                throw new ConfigException("Failed to load ca config for iam", e);
            }

        } else {
            String privateKeyStr =
                config.getString("private-key", null);
            if (privateKeyStr == null) {
                privateKey = defaults.privateKey();
            } else {
                try {
                    KeyFactory keyFactory = KeyFactory.getInstance("RSA");
                    this.privateKey = keyFactory.generatePrivate(
                        new PKCS8EncodedKeySpec(
                            Base64.getDecoder().decode(privateKeyStr)));
                } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                    throw new ConfigException("Failed to load ca config for iam", e);
                }
            }

            this.serviceAccountId =
                config.getString("service-account-id", defaults.serviceAccountId());
            this.keyId = config.getString("key-id", defaults.keyId());
        }

        this.audience = config.getString("audience");
        this.database = config.getString("database");
        this.discoveryEndpoint = config.getString("endpoint");
    }

    @Override()
    public PrivateKey privateKey() {
        return privateKey;
    }

    public T privateKey(final PrivateKey value) {
        this.privateKey = value;
        return self();
    }

    @Override()
    public String serviceAccountId() {
        return serviceAccountId;
    }

    public T serviceAccountId(final String value) {
        this.serviceAccountId = value;
        return self();
    }

    @Override()
    public String keyId() {
        return keyId;
    }

    public T keyId(final String value) {
        this.keyId = value;
        return self();
    }

    @Override
    public String discoveryEndpoint() {
        return discoveryEndpoint;
    }

    public T discoveryEndpoint(final String discoveryEndpoint) {
        this.discoveryEndpoint = discoveryEndpoint;
        return self();
    }

    @Override
    public String database() {
        return database;
    }

    public T database(final String database) {
        this.database = database;
        return self();
    }

    @Override
    public String audience() {
        return audience;
    }

    public T audience(final String audience) {
        this.audience = audience;
        return self();
    }
}
