package ru.yandex.logbroker2.config;

import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import ru.yandex.http.config.URIConfig;
import ru.yandex.http.config.URIConfigBuilder;
import ru.yandex.logbroker2.LBDataFormat;
import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;

public abstract class AbstractLogbroker2SingleConsumerConfigBuilder
    <T extends AbstractLogbroker2SingleConsumerConfigBuilder<T>>
    implements ConfigBuilder<T>, Logbroker2SingleConsumerConfig
{
    private static final String SEND_BATCH_SIZE = "send-batch-size";

    private URIConfigBuilder targetConfig;
    private Set<String> topics;
    private String clientId;
    private Set<String> balancerHosts;
    private int balancerPort;
    private int proxyPort;
    private int prefetchCount;
    private long prefetchDataSize;
    private int perTopicPrefetchCount;
    private long perTopicPrefetchDataSize;
    private Map<String, Integer> sendBatchSize;
    private boolean readOnlyLocal;
    private boolean gzipRequests;
    private String tvmClientId;
    private IamJwtConfigBuilder iamConfig;
    private LBDataFormat dataFormat;
    private Map<String, FieldConfigBuilder> fieldsConfig;

    protected AbstractLogbroker2SingleConsumerConfigBuilder(
        final Logbroker2SingleConsumerConfig config)
    {
        targetConfig = new URIConfigBuilder(config.targetConfig());
        topics = config.topics();
        clientId = config.clientId();
        balancerHosts = config.balancerHosts();
        balancerPort = config.balancerPort();
        proxyPort = config.proxyPort();
        prefetchCount = config.prefetchCount();
        prefetchDataSize = config.prefetchDataSize();
        perTopicPrefetchCount = config.perTopicPrefetchCount();
        perTopicPrefetchDataSize = config.perTopicPrefetchDataSize();
        sendBatchSize = config.sendBatchSize();
        tvmClientId = config.tvmClientId();

        if (config.iamConfig() != null) {
            this.iamConfig = new IamJwtConfigBuilder(config.iamConfig());
        } else {
            this.iamConfig = null;
        }

        readOnlyLocal = config.readOnlyLocal();
        gzipRequests = config.gzipRequests();
        dataFormat = config.dataFormat();
        fieldsConfig(config.fieldsConfig());
    }

    protected AbstractLogbroker2SingleConsumerConfigBuilder(
        final IniConfig config,
        final Logbroker2SingleConsumerConfig defaults)
        throws ConfigException
    {
        targetConfig = new URIConfigBuilder(
            config.section("target"),
            defaults.targetConfig());
        topics = config.get(
            "topics",
            Collections.emptySet(),
            new CollectionParser<>(
                NonEmptyValidator.INSTANCE,
                LinkedHashSet::new));
        clientId = config.getString("client-id");
        balancerHosts = config.get(
            "balancer-hosts",
            Collections.emptySet(),
            new CollectionParser<>(
                NonEmptyValidator.INSTANCE,
                LinkedHashSet::new));
        balancerPort = config.get(
            "balancer-port",
            defaults.balancerPort(),
            NonNegativeIntegerValidator.INSTANCE);
        proxyPort = config.get(
            "proxy-port",
            defaults.proxyPort(),
            NonNegativeIntegerValidator.INSTANCE);
        prefetchCount = config.get(
            "prefetch-count",
            defaults.prefetchCount(),
            NonNegativeIntegerValidator.INSTANCE);
        prefetchDataSize = config.get(
            "prefetch-data-size",
            defaults.prefetchDataSize(),
            LongMemorySizeParser.INSTANCE);
        perTopicPrefetchCount = config.get(
            "per-topic-prefetch-count",
            defaults.perTopicPrefetchCount(),
            NonNegativeIntegerValidator.INSTANCE);
        perTopicPrefetchDataSize = config.get(
            "per-topic-prefetch-data-size",
            defaults.perTopicPrefetchDataSize(),
            LongMemorySizeParser.INSTANCE);
        sendBatchSize = new HashMap<>();
        sendBatchSize.put(
            null,
            config.get(
                SEND_BATCH_SIZE,
                defaults.sendBatchSize().get(null),
                NonNegativeIntegerValidator.INSTANCE));
        IniConfig sendBatchSizeSection = config.section(SEND_BATCH_SIZE);
        for (String key: sendBatchSizeSection.keys()) {
            int batchSize = sendBatchSizeSection.get(
                key,
                NonNegativeIntegerValidator.INSTANCE);
            sendBatchSize.put(key, batchSize);
        }

        readOnlyLocal = config.getBoolean(
            "read-only-local",
            defaults.readOnlyLocal());
        gzipRequests = config.getBoolean(
            "gzip-requests",
            defaults.gzipRequests());

        tvmClientId =
            config.getString("tvm-client-id", defaults.tvmClientId());
        IniConfig iamSection = config.sectionOrNull("iam");
        if (iamSection != null) {
            this.iamConfig = new IamJwtConfigBuilder(iamSection, defaults.iamConfig());
        } else {
            this.iamConfig = null;
        }
        dataFormat = config.get(
            "data-format",
            defaults.dataFormat(),
            new EnumParser<>(LBDataFormat.class));

        Map<String, IniConfig> fieldsSections =
            config.section("field").sections();
        fieldsConfig = new LinkedHashMap<>(fieldsSections.size() << 1);
        for (Map.Entry<String, IniConfig> entry: fieldsSections.entrySet()) {
            fieldsConfig.put(
                entry.getKey(),
                new FieldConfigBuilder(entry.getValue()));
        }
    }

    @Override
    public URIConfigBuilder targetConfig() {
        return targetConfig;
    }

    public T targetConfig(final URIConfig targetConfig) {
        this.targetConfig = new URIConfigBuilder(targetConfig);
        return self();
    }

    @Override
    public Set<String> balancerHosts() {
        return balancerHosts;
    }

    public T balancerHosts(final Set<String> balancerHosts) {
        this.balancerHosts = balancerHosts;
        return self();
    }

    @Override
    public int balancerPort() {
        return balancerPort;
    }

    public T balancerPort(final int balancerPort) {
        this.balancerPort = balancerPort;
        return self();
    }

    @Override
    public int proxyPort() {
        return proxyPort;
    }

    public T proxyPort(final int proxyPort) {
        this.proxyPort = proxyPort;
        return self();
    }

    @Override
    public int prefetchCount() {
        return prefetchCount;
    }

    public T prefetchCount(final int prefetchCount) {
        this.prefetchCount = prefetchCount;
        return self();
    }

    @Override
    public long prefetchDataSize() {
        return prefetchDataSize;
    }

    public T prefetchDataSize(final long prefetchDataSize) {
        this.prefetchDataSize = prefetchDataSize;
        return self();
    }

    @Override
    public int perTopicPrefetchCount() {
        return perTopicPrefetchCount;
    }

    public T perTopicPrefetchCount(final int perTopicPrefetchCount) {
        this.perTopicPrefetchCount = perTopicPrefetchCount;
        return self();
    }

    @Override
    public long perTopicPrefetchDataSize() {
        return perTopicPrefetchDataSize;
    }

    public T perTopicPrefetchDataSize(final long perTopicPrefetchDataSize) {
        this.perTopicPrefetchDataSize = perTopicPrefetchDataSize;
        return self();
    }

    @Override
    public Map<String, Integer> sendBatchSize() {
        return sendBatchSize;
    }

    public T sendBatchSize(final Map<String, Integer> sendBatchSize) {
        this.sendBatchSize = new HashMap<>(sendBatchSize);
        return self();
    }

    @Override
    public Set<String> topics() {
        return topics;
    }

    public T topics(final Set<String> topics) {
        this.topics = topics;
        return self();
    }

    @Override
    public String clientId() {
        return clientId;
    }

    public T clientId(final String clientId) {
        this.clientId = clientId;
        return self();
    }

    @Override
    public boolean readOnlyLocal() {
        return readOnlyLocal;
    }

    public T readOnlyLocal(final boolean readOnlyLocal) {
        this.readOnlyLocal = readOnlyLocal;
        return self();
    }

    @Override
    public boolean gzipRequests() {
        return gzipRequests;
    }

    public T gzipRequests(final boolean gzipRequests) {
        this.gzipRequests = gzipRequests;
        return self();
    }

    @Override
    public String tvmClientId() {
        return tvmClientId;
    }

    public T tvmClientId(final String tvmClientId) {
        this.tvmClientId = tvmClientId;
        return self();
    }

    @Override
    public IamJwtConfigBuilder iamConfig() {
        return iamConfig;
    }

    public T iamConfig(
        final IamJwtConfig iamConfig)
    {
        if (iamConfig != null) {
            this.iamConfig = new IamJwtConfigBuilder(iamConfig);
        } else {
            this.iamConfig = null;
        }

        return self();
    }

    @Override
    public LBDataFormat dataFormat() {
        return dataFormat;
    }

    public T dataFormat(final LBDataFormat dataFormat) {
        this.dataFormat = dataFormat;
        return self();
    }

    @Override
    public Map<String, FieldConfigBuilder> fieldsConfig() {
        return fieldsConfig;
    }

    public T fieldsConfig(
        final Map<String, ? extends FieldConfig> fieldsConfig)
    {
        this.fieldsConfig = new LinkedHashMap<>(fieldsConfig.size() << 1);
        for (Map.Entry<String, ? extends FieldConfig> entry
            : fieldsConfig.entrySet())
        {
            this.fieldsConfig.put(
                entry.getKey(),
                new FieldConfigBuilder(entry.getValue()));
        }
        return self();
    }
}

