package ru.yandex.logbroker2.config;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.logbroker2.LBDataFormat;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.ImmutableConfig;

public class ImmutableLogbroker2SingleConsumerConfig
    implements ImmutableConfig, Logbroker2SingleConsumerConfig
{
    private final ImmutableURIConfig targetConfig;
    private final Set<String> topics;
    private final String clientId;
    private final Set<String> balancerHosts;
    private final int balancerPort;
    private final int proxyPort;
    private final int prefetchCount;
    private final long prefetchDataSize;
    private final int perTopicPrefetchCount;
    private final long perTopicPrefetchDataSize;
    private final Map<String, Integer> sendBatchSize;
    private final boolean readOnlyLocal;
    private final boolean gzipRequests;
    private final String tvmClientId;
    private final ImmutableIamJwtConfig iamConfig;
    private final LBDataFormat dataFormat;
    private final Map<String, ImmutableFieldConfig> fieldsConfig;

    public ImmutableLogbroker2SingleConsumerConfig(
        final Logbroker2SingleConsumerConfig config)
        throws ConfigException
    {
        targetConfig = new ImmutableURIConfig(config.targetConfig());
        topics = config.topics();
        clientId = config.clientId();
        balancerHosts = config.balancerHosts();
        balancerPort = config.balancerPort();
        proxyPort = config.proxyPort();
        prefetchCount = config.prefetchCount();
        prefetchDataSize = config.prefetchDataSize();
        perTopicPrefetchCount = config.perTopicPrefetchCount();
        perTopicPrefetchDataSize = config.perTopicPrefetchDataSize();
        sendBatchSize = Collections.unmodifiableMap(config.sendBatchSize());
        readOnlyLocal = config.readOnlyLocal();
        gzipRequests = config.gzipRequests();
        tvmClientId = config.tvmClientId();
        IamJwtConfig iamConfig = config.iamConfig();
        if (tvmClientId != null && iamConfig != null) {
            throw new ConfigException(
                "Both tvm-client-id and iam configured. "
                + "Only one authentication protocol allowed");
        }
        if (iamConfig != null) {
            this.iamConfig = new ImmutableIamJwtConfig(iamConfig);
        } else {
            this.iamConfig = null;
        }
        dataFormat = config.dataFormat();
        Map<String, ? extends FieldConfig> fieldsConfig =
            config.fieldsConfig();
        this.fieldsConfig = new LinkedHashMap<>(fieldsConfig.size() << 1);
        for (Map.Entry<String, ? extends FieldConfig> entry
            : fieldsConfig.entrySet())
        {
            this.fieldsConfig.put(
                entry.getKey(),
                new ImmutableFieldConfig(entry.getValue()));
        }
        if (topics.size() == 0) {
            throw new ConfigException("topics collection is not set");
        }
        if (clientId == null) {
            throw new ConfigException("clientId is not set");
        }
        if (balancerHosts.isEmpty()) {
            throw new ConfigException("balancer-hosts is not set");
        }
        if (dataFormat == LBDataFormat.RAW) {
            if (!fieldsConfig.isEmpty()) {
                throw new ConfigException(
                    "Data format is RAW, but some fields configured");
            }
        } else if (fieldsConfig.isEmpty()) {
            throw new ConfigException(
                "Data format is not RAW, but no fields configured");
        }
    }

    @Override
    public ImmutableURIConfig targetConfig() {
        return targetConfig;
    }

    @Override
    public Set<String> balancerHosts() {
        return balancerHosts;
    }

    @Override
    public int balancerPort() {
        return balancerPort;
    }

    @Override
    public int proxyPort() {
        return proxyPort;
    }

    @Override
    public int prefetchCount() {
        return prefetchCount;
    }

    @Override
    public long prefetchDataSize() {
        return prefetchDataSize;
    }

    @Override
    public int perTopicPrefetchCount() {
        return perTopicPrefetchCount;
    }

    @Override
    public long perTopicPrefetchDataSize() {
        return perTopicPrefetchDataSize;
    }

    @Override
    public Map<String, Integer> sendBatchSize() {
        return sendBatchSize;
    }

    @Override
    public Set<String> topics() {
        return topics;
    }

    @Override
    public String clientId() {
        return clientId;
    }

    @Override
    public boolean readOnlyLocal() {
        return readOnlyLocal;
    }

    @Override
    public boolean gzipRequests() {
        return gzipRequests;
    }

    @Override
    public String tvmClientId() {
        return tvmClientId;
    }

    @Override
    public ImmutableIamJwtConfig iamConfig() {
        return iamConfig;
    }

    @Override
    public LBDataFormat dataFormat() {
        return dataFormat;
    }

    @Override
    public Map<String, ImmutableFieldConfig> fieldsConfig() {
        return fieldsConfig;
    }
}

