package ru.yandex.logbroker;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.json.simple.parser.JSONParser;
import org.json.simple.parser.ParseException;

import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.logbroker.config.ImmutableLogConfig;
import ru.yandex.logbroker.config.ImmutableLogbrokerConsumerConfig;
import ru.yandex.logbroker.config.ImmutableLogbrokerConsumerServerConfig;
import ru.yandex.logbroker.log.LogConsumeManager;
import ru.yandex.logbroker.log.consumer.LogConsumerFactory;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.StatsConsumer;

public class LogbrokerConsumer
    extends AbstractStatable
    implements Stoppable, Closeable, DcUpdateListener
{
    private static final String NAME = "LogbrokerConsumer";

    private static final int DC_NAME_LENGTH = 3;
    private static final String HOST_PROP_NAME = "BSCONFIG_IHOST";
    private static final String DC_TAG_PREFIX = "a_dc_";

    private final GenericAutoCloseableChain<IOException> closeChain =
        new GenericAutoCloseableChain<>();
    private final Map<String, LogConsumeManager> logConsumers =
        new HashMap<>();
    private final PrefixedLogger logger;
    private final ImmutableLogbrokerConsumerConfig consumerConfig;
    private final LogbrokerDcUpdater dcUpdater;

    public LogbrokerConsumer(
        final SharedConnectingIOReactor reactor,
        final ImmutableLogbrokerConsumerServerConfig config,
        final PrefixedLogger logger)
        throws ConfigException
    {
        this.logger = logger.addPrefix(NAME);
        consumerConfig = config.consumerConfig();
        String dc = currentDc(consumerConfig);

        for (Map.Entry<String, ImmutableLogConfig> entry
            : consumerConfig.logConfig().entrySet())
        {
            LogConsumeManager manager =
                new LogConsumeManager(
                    entry.getValue(),
                    reactor,
                    config,
                    logger,
                    dc);
            logConsumers.put(entry.getKey(), manager);
            closeChain.add(manager);
        }

        dcUpdater = new LogbrokerDcUpdater(
            this,
            logger,
            consumerConfig.logbrokerHostConfig(),
            config.dnsConfig());
        closeChain.add(dcUpdater);
    }

    public List<LogConsumerFactory> consumers() {
        return logConsumers.values().stream()
            .map(LogConsumeManager::consumerFactory)
            .collect(Collectors.toList());
    }

    @Override
    public void removeDc(final LogbrokerDc dc) {
        logConsumers.values().forEach(
            m -> m.removeDc(dc, consumerConfig.stopWaitTimeout()));
    }

    @Override
    public void addDc(final LogbrokerDc dc) {
        logConsumers.values().forEach(m -> m.addDc(dc));
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        for (LogConsumeManager consumer: logConsumers.values()) {
            consumer.stats(statsConsumer);
        }
    }

    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> result = new LinkedHashMap<>();
        final Map<String, Object> logStatus = new LinkedHashMap<>();
        for (Map.Entry<String, LogConsumeManager> m: logConsumers.entrySet()) {
            logStatus.put(m.getKey(), m.getValue().status(verbose));
        }

        result.put(NAME, logStatus);
        return result;
    }

    public void start() {
        logConsumers.values().forEach(LogConsumeManager::start);
        dcUpdater.start();
    }

    @Override
    public void stop() {
        dcUpdater.stop();
        logConsumers.values().forEach(Stoppable::stop);
    }

    @Override
    public boolean waitStop(final long deadline) throws InterruptedException {
        boolean result = true;
        for (Stoppable logConsumer: logConsumers.values()) {
            result &= logConsumer.waitStop(deadline);
        }

        result &= dcUpdater.waitStop(deadline);
        if (!result) {
            logger.warning("Failed to gracefully stop, sayonara");
        } else {
            logger.info("Stopped");
        }

        return result;
    }

    @Override
    public boolean stopped() {
        return true;
    }

    @Override
    public void close() throws IOException {
        logger.warning("Closing, trying to gracefully stop");
        stop();
        logger.warning(
            "Stop requested, waiting for " + consumerConfig.stopWaitTimeout()
            + " ms");
        try {
            waitStop(
                System.currentTimeMillis() + consumerConfig.stopWaitTimeout());
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Waiting stop was interrupted", ie);
        }

        this.closeChain.close();
    }

    private static String currentDc(
        final ImmutableLogbrokerConsumerConfig config)
    {
        String consumerDc;

        if (config.dc() != null) {
            consumerDc = config.dc();
        } else {
            JSONParser parser = new JSONParser();

            try (FileInputStream fis =
                     new FileInputStream(
                         new File(config.dumpJsonPath().trim()));
                 Reader reader =
                     new InputStreamReader(fis, Charset.defaultCharset()))
            {
                Map<?, ?> dump = ValueUtils.asMap(parser.parse(reader));
                consumerDc = parseDcFromJson(dump);
            } catch (
                ParseException | IOException | JsonUnexpectedTokenException e)
            {
                throw new RuntimeException(e);
            }
        }

        return consumerDc;
    }

    public static String parseDcFromJson(final Map<?, ?> dump)
        throws JsonUnexpectedTokenException
    {
        Map<?, ?> props = ValueUtils.asMap(dump.get("properties"));
        String tagsStr = ValueUtils.asString(props.get("tags"));

        String[] tags = tagsStr.split("\\s+");
        for (String tag : tags) {
            if (tag.startsWith(DC_TAG_PREFIX)) {
                return tag.substring(DC_TAG_PREFIX.length(), tag.length());
            }
        }

        String host = ValueUtils.asString(props.get(HOST_PROP_NAME));
        if (host.length() < DC_NAME_LENGTH) {
            throw new RuntimeException(
                "To short host in json " + host);
        }

        return host.substring(0, DC_NAME_LENGTH);
    }
}
