package ru.yandex.logbroker;

import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpHost;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.logbroker.client.MultidcLogbrokerClient;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.StaterConfigBuilder;

public class LogbrokerDcUpdater
    extends AbstractStatable
    implements Runnable, Stoppable, GenericAutoCloseable<IOException>
{
    private static final String NAME = "DcUpdater";
    private static final int INTERVAL = 300;
    private static final int DC_UPDATE_INTERVAL = 1000 * 60 * 5;
    private static final int DC_EXPIRE_TIME = 1000 * 60 * 60;

    private final Thread thread;

    private final MultidcLogbrokerClient client;
    private final PrefixedLogger logger;
    private final Map<String, DcEntry> dcMap;
    private final DcUpdateListener listener;
    private volatile boolean stop = false;
    private long lastDcFetch = 0;

    // CSOFF: ParameterNumber
    public LogbrokerDcUpdater(
        final DcUpdateListener listener,
        final PrefixedLogger logger,
        final ImmutableHttpHostConfig rootLbClientConfig,
        final ImmutableDnsConfig dnsConfig)
    {
        this.listener = listener;

        RequestsStater clientStater =
            new RequestsStater(
                new StaterConfigBuilder().prefix("/lbclient_global").build());

        client = new MultidcLogbrokerClient(
                rootLbClientConfig,
                dnsConfig,
                clientStater,
                logger);

        registerStater(clientStater);

        this.dcMap = new LinkedHashMap<>();
        this.logger = logger.addPrefix(NAME);
        this.thread = new Thread(new ThreadGroup("Thread-" + NAME), this);
    }
    // CSON: ParameterNumber

    public void start() {
        this.thread.start();
    }

    @Override
    public void close() throws IOException {
        thread.interrupt();
    }

    @Override
    public void stop() {
        stop = true;
    }

    @Override
    public boolean stopped() {
        return !thread.isAlive();
    }

    @Override
    public void run() {
        while (!stop) {
            try {
                long now = System.currentTimeMillis();
                fetchDc(now);
                checkStale(now);

                Thread.sleep(INTERVAL);
            } catch (InterruptedException ie) {
                logger.log(Level.WARNING, "Dc updater interrupted", ie);
                break;
            }
        }
    }

    private void checkStale(final long now) {
        Iterator<Map.Entry<String, DcEntry>> iterator =
            dcMap.entrySet().iterator();
        while (iterator.hasNext()) {
            DcEntry entry = iterator.next().getValue();
            if (now - entry.lastUpdateTime > DC_EXPIRE_TIME) {
                logger.warning(
                    "DC expired last update "
                        + entry.lastUpdateTime
                        + " now " + now);
                iterator.remove();
                listener.removeDc(entry.dc());
            }
        }
    }

    private void fetchDc(final long now) {
        if (now - lastDcFetch < DC_UPDATE_INTERVAL) {
            return;
        }

        try {
            Map<String, HttpHost> dcs = client.clusters();
            lastDcFetch = now;

            for (Map.Entry<String, HttpHost> dc: dcs.entrySet()) {
                DcEntry entry = dcMap.get(dc.getKey());
                if (entry != null) {
                    if (entry.dc().balancer().equals(dc.getValue())) {
                        entry.update();
                        continue;
                    }

                    logger.warning(
                        "Dc " + entry.dc().name()
                            + " update balancer from "
                            + entry.dc().balancer().toHostString()
                            + " to " + dc.getValue().toHostString());
                    listener.removeDc(entry.dc());
                }

                entry = new DcEntry(
                    new LogbrokerDc(dc.getKey(), dc.getValue()),
                    now);
                dcMap.put(entry.dc().name(), entry);
                listener.addDc(entry.dc());
            }
        } catch (LogbrockerClientException lbe) {
            logger.log(Level.WARNING, "Failed to  retrieve dc list", lbe);
        }
    }

    private static final class DcEntry {
        private final LogbrokerDc dc;
        private long lastUpdateTime;

        private DcEntry(final LogbrokerDc dc) {
            this(dc, System.currentTimeMillis());
        }

        private DcEntry(final LogbrokerDc dc, final long now) {
            this.dc = dc;
            this.lastUpdateTime = now;
        }

        public LogbrokerDc dc() {
            return dc;
        }

        public void update() {
            this.lastUpdateTime = System.currentTimeMillis();
        }
    }
}
