package ru.yandex.logbroker.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.client.exception.ParseException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public abstract class AbstractLogbrokerClient implements LogbrokerClient {
    public static final String CLIENT = "client";
    protected static final String LIST = "/pull/list?";
    protected static final String OFFSETS = "/pull/offsets?";

    protected final ImmutableDnsConfig dnsConfig;
    protected final RequestsStater stater;
    protected final PrefixedLogger logger;
    protected final CloseableHttpClient client;

    protected AbstractLogbrokerClient(
        final ImmutableHttpTargetConfig config,
        final ImmutableDnsConfig dnsConfig,
        final RequestsStater stater,
        final PrefixedLogger logger)
    {
        this.stater = stater;
        this.dnsConfig = dnsConfig;

        if (logger != null) {
            this.logger = logger.addPrefix("LB_CLIENT");
        } else {
            this.logger = new PrefixedLogger(
                Logger.getAnonymousLogger(), "", " ");
        }

        client = ClientBuilder.createClient(config, dnsConfig);
    }

    @Override
    public List<String> list(
        final HttpHost host,
        final String ident,
        final String logType)
        throws LogbrockerClientException
    {
        return this.list(host, ident, logType, true);
    }

    public List<String> list(
        final HttpHost host,
        final String ident,
        final String logType,
        final boolean dcLocal)
        throws LogbrockerClientException
    {
        List<String> result = new ArrayList<>();
        QueryConstructor uri = new QueryConstructor(LIST);
        try {
            if (dcLocal) {
                uri.append(DC, DC_LOCAL);
            }

            uri.append(IDENT, ident);
            uri.append(LOG_TYPE, logType);
        } catch (BadRequestException bre) {
            throw new LogbrockerClientException(bre);
        }

        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, uri.toString());
        if (logger != null) {
            logger.fine("List request from " + host + ' ' + request);
        }

        try (CloseableHttpResponse response = execute(host, request)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            }

            String responseData = CharsetUtils.toString(response.getEntity());
            List<Map<String, String>> data =
                LogbrokerClientResponseParser.parse(TOPIC, responseData);

            data.forEach(map -> result.add(map.get(TOPIC)));
        } catch (IOException | HttpException | ParseException e) {
            throw new LogbrockerClientException(e);
        }

        return result;
    }

    public static long extractContentLength(
        final CloseableHttpResponse response)
    {
        HttpEntity entity = response.getEntity();
        long responseLength;
        if (entity == null) {
            responseLength = 0L;
        } else {
            responseLength = Math.max(0L, entity.getContentLength());
        }
        return responseLength;
    }

    protected CloseableHttpResponse execute(
        final HttpHost host,
        final HttpRequest request)
        throws IOException
    {
        long start = System.currentTimeMillis();

        try {
            CloseableHttpResponse response = client.execute(host, request);
            stater.accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    response.getStatusLine().getStatusCode(),
                    start,
                    start,
                    0L,
                    extractContentLength(response)));
            return response;
        } catch (IOException e) {
            stater.accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST,
                    start,
                    start,
                    0L,
                    0L));
            throw e;
        }
    }

    public List<OffsetInfo> offsets(
        final HttpHost host,
        final String clientId,
        final String topic)
        throws LogbrockerClientException
    {
        List<OffsetInfo> result = new ArrayList<>();
        List<Map<String, String>> offsetList =
            offsetsList(host, clientId, topic);

        try {
            for (Map<String, String> data: offsetList) {
                result.add(BasicOffsetInfo.fromMap(data));
            }
        } catch (ParseException pe) {
            throw new LogbrockerClientException(pe);
        }

        return result;
    }

    protected List<Map<String, String>> offsetsList(
        final HttpHost host,
        final String clientId,
        final String topic)
        throws LogbrockerClientException
    {
        QueryConstructor uri = new QueryConstructor(OFFSETS);
        try {
            uri.append(CLIENT, clientId);
            uri.append(TOPIC, topic);
        } catch (BadRequestException bre) {
            throw new LogbrockerClientException(bre);
        }

        logger.fine("Offset request from " + host + ' ' + uri.toString());
        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, uri.toString());
        try (CloseableHttpResponse response = execute(host, request)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            }

            return LogbrokerClientResponseParser.parse(response);
        } catch (IOException | HttpException | ParseException e) {
            throw new LogbrockerClientException(e);
        }
    }
}
