package ru.yandex.logbroker.client;

import java.util.Map;

import org.apache.http.HttpHost;

import ru.yandex.logbroker.client.exception.ParseException;

public class BasicPartition implements Partition {
    private static final String ID = "id";
    private static final String HOST = "host";
    private static final String PORT = "port";
    private static final String TOPIC = "topic";

    private final String id;
    private final HttpHost host;
    private final String topic;
    private final String idString;

    // CSOFF: ParameterNumber
    public BasicPartition(
        final String id,
        final String host,
        final int port,
        final String topic)
    {
        this.id = id;
        if (host == null) {
            this.host = null;
        } else {
            if (port == -1) {
                this.host = new HttpHost(host);
            } else {
                this.host = new HttpHost(host, port);
            }
        }

        this.topic = topic;
        this.idString = topic + ':' + id;
    }
    // CSON: ParameterNumber

    public static BasicPartition fromMap(final Map<String, String> map)
        throws ParseException
    {
        int port = -1;
        try {
            if (map.containsKey(PORT)) {
                port = Integer.parseInt(map.get(PORT));
            }
        } catch (NumberFormatException nfe) {
            throw new ParseException("Not valid port " + map.toString(), nfe);
        }

        String id = map.get(ID);
        if (id == null) {
            throw new ParseException(
                "No partition id was found" + map.toString());
        }

        String host = map.get(HOST);
        if (host == null) {
            throw new ParseException(
                "No partition host was found" + map.toString());
        }

        String topic = map.get(TOPIC);
        if (topic == null) {
            throw new ParseException(
                "No partition topic was found" + map.toString());
        }

        return new BasicPartition(id, host, port, topic);
    }

    @Override
    public String id() {
        return id;
    }

    @Override
    public HttpHost host() {
        return host;
    }

    @Override
    public String topic() {
        return topic;
    }

    @Override
    public String idString() {
        return idString;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        if (host != null) {
            sb.append(host.toHostString());
        }

        sb.append(':');
        sb.append(idString);
        return sb.toString();
    }
}
