package ru.yandex.logbroker.client;

import java.io.IOException;
import java.io.InputStream;

import org.apache.http.ConnectionClosedException;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.MalformedChunkCodingException;
import org.apache.http.TruncatedChunkException;
import org.apache.http.config.MessageConstraints;
import org.apache.http.impl.io.AbstractMessageParser;
import org.apache.http.io.BufferInfo;
import org.apache.http.io.SessionInputBuffer;
import org.apache.http.util.Args;
import org.apache.http.util.CharArrayBuffer;

import ru.yandex.logger.PrefixedLogger;

public class ChunkedInputStream extends InputStream {
    private static final int SKIP_BUFFER = 4096;
    private static final int CHUNK_LEN = 1;
    private static final int CHUNK_DATA = 2;
    private static final int CHUNK_CRLF = 3;
    private static final int CHUNK_INVALID = 2147483647;
    private static final int CHAR_BUFFER_SIZE = 16;

    private final SessionInputBuffer in;
    private final CharArrayBuffer buffer;
    private final MessageConstraints constraints;
    private final PrefixedLogger logger;
    private int state;
    private long chunkSize;
    private long pos;
    private boolean eof;
    private boolean closed;
    private Header[] footers;

    public ChunkedInputStream(
        final SessionInputBuffer in,
        final MessageConstraints constraints,
        final PrefixedLogger logger)
    {
        this.logger = logger;

        this.eof = false;
        this.closed = false;
        this.footers = new Header[0];
        this.in = Args.notNull(in, "Session input buffer");
        this.pos = 0L;
        this.buffer = new CharArrayBuffer(CHAR_BUFFER_SIZE);
        this.state = CHUNK_LEN;

        if (constraints != null) {
            this.constraints = constraints;
        } else {
            this.constraints = MessageConstraints.DEFAULT;
        }
    }

    public ChunkedInputStream(final SessionInputBuffer in) {
        this(in, null, null);
    }

    @Override
    public int available() throws IOException {
        if (this.in instanceof BufferInfo) {
            int len = ((BufferInfo) this.in).length();
            return (int) Math.min((long) len, this.chunkSize - this.pos);
        } else {
            return 0;
        }
    }

    // CSOFF: ReturnCount
    public byte[] readChunk() throws IOException {
        if (this.closed) {
            throw new IOException("Attempted read from closed stream.");
        } else if (this.eof) {
            return null;
        } else {
            if (this.state != CHUNK_DATA) {
                this.nextChunk();
                if (this.eof) {
                    return null;
                }
            }

            int len = (int) (this.chunkSize - this.pos);
            byte[] data = new byte[len];
            int bytesRead = 0;
            while (bytesRead < len) {
                bytesRead += this.in.read(data, bytesRead, len - bytesRead);
            }

            if (bytesRead != -1) {
                this.pos += (long) bytesRead;
                if (this.pos >= this.chunkSize) {
                    this.state = CHUNK_CRLF;
                }

                return data;
            } else {
                this.eof = true;
                throw new TruncatedChunkException(
                    "Truncated chunk  ( expected size: "
                        + this.chunkSize + ";  actual size: "
                        + this.pos
                        + " )");
            }
        }
    }

    @Override
    public int read() throws IOException {
        if (this.closed) {
            throw new IOException("Attempted read from closed stream");
        } else if (this.eof) {
            return -1;
        } else {
            if (this.state != CHUNK_DATA) {
                this.nextChunk();
                if (this.eof) {
                    return -1;
                }
            }

            int b = this.in.read();
            if (b != -1) {
                ++this.pos;
                if (this.pos >= this.chunkSize) {
                    this.state = CHUNK_CRLF;
                }
            }

            return b;
        }
    }

    @Override
    public int read(
        final byte[] b,
        final int off,
        final int len)
        throws IOException
    {
        if (this.closed) {
            throw new IOException("Attempted read from closed stream ");
        } else if (this.eof) {
            return -1;
        } else {
            if (this.state != CHUNK_DATA) {
                this.nextChunk();
                if (this.eof) {
                    return -1;
                }
            }

            int bytesRead =
                this.in.read(
                    b,
                    off,
                    (int) Math.min((long) len, this.chunkSize - this.pos));
            if (bytesRead != -1) {
                this.pos += (long) bytesRead;
                if (this.pos >= this.chunkSize) {
                    this.state = CHUNK_CRLF;
                }

                return bytesRead;
            } else {
                this.eof = true;
                throw new TruncatedChunkException(
                    "Truncated chunk ( expected size: "
                        + this.chunkSize + "; actual size: " + this.pos + ')');
            }
        }
    }
    // CSON: ReturnCount

    @Override
    public int read(final byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    private void nextChunk() throws IOException {
        if (this.state == CHUNK_INVALID) {
            throw new MalformedChunkCodingException("Corrupt data stream");
        } else {
            try {
                this.chunkSize = this.getChunkSize();
                if (this.chunkSize < 0L) {
                    throw new MalformedChunkCodingException(
                        "Negative chunk size");
                } else {
                    this.state = CHUNK_DATA;
                    this.pos = 0L;
                    if (this.chunkSize == 0L) {
                        this.eof = true;
                        this.parseTrailerHeaders();
                    }
                }
            } catch (MalformedChunkCodingException mcce) {
                this.state = CHUNK_INVALID;
                throw mcce;
            }
        }
    }

    //CSOFF: FallThrough
    @SuppressWarnings("fallthrough")
    private long getChunkSize() throws IOException {
        int st = this.state;
        switch (st) {
            case CHUNK_CRLF:
                this.buffer.clear();
                int bytesRead1 = this.in.readLine(this.buffer);
                if (bytesRead1 == -1) {
                    throw new MalformedChunkCodingException(
                        "CRLF expected at end of chunk");
                } else if (!this.buffer.isEmpty()) {
                    throw new MalformedChunkCodingException(
                        "Unexpected content at the end of chunk");
                } else {
                    this.state = CHUNK_LEN;
                }
            case CHUNK_LEN:
                this.buffer.clear();
                int bytesRead2 = this.in.readLine(this.buffer);
                if (bytesRead2 == -1) {
                    throw new ConnectionClosedException(
                        "Premature end of chunk coded message body: "
                            + "closing chunk expected");
                } else {
                    int separator = this.buffer.indexOf(';');
                    if (separator < 0) {
                        separator = this.buffer.length();
                    }

                    String s = this.buffer.substringTrimmed(0, separator);

                    try {
                        return Long.parseLong(s, CHAR_BUFFER_SIZE);
                    } catch (NumberFormatException nfe) {
                        throw new MalformedChunkCodingException(
                            "Bad chunk header: " + s);
                    }
                }
            default:
                throw new IllegalStateException("Inconsistent codec state");
        }
    }
    //CSON: FallThrough

    private void parseTrailerHeaders() throws IOException {
        try {
            this.footers =
                AbstractMessageParser.parseHeaders(
                    this.in,
                    this.constraints.getMaxHeaderCount(),

                    this.constraints.getMaxLineLength(),
                    null);
        } catch (HttpException he) {
            MalformedChunkCodingException ioe =
                new MalformedChunkCodingException(
                    "Invalid footer: " + he.getMessage());
            ioe.initCause(he);
            throw ioe;
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            long wasted = 0;
            try {
                if (!this.eof && this.state != CHUNK_INVALID) {
                    byte[] buff = new byte[SKIP_BUFFER];

                    // read and discard the remainder of the message
                    while (read(buff) >= 0) {
                        continue;
                    }
                }
            } finally {
                this.eof = true;
                this.closed = true;
                if (this.logger != null) {
                    this.logger.warning(
                        "Wasted on stream close bytes: " + wasted);
                }
            }
        }
    }

    public Header[] getFooters() {
        return this.footers.clone();
    }
}
