package ru.yandex.logbroker.client;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.security.AccessController;
import java.security.PrivilegedAction;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.conn.EofSensorInputStream;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.util.EntityUtils;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.client.exception.LogbrokerConnectionsConflictException;
import ru.yandex.logbroker.config.ClientConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class ChunkedSession implements Session {
    static final String CLIENT = "client";
    static final String TOPIC = "topic";
    static final String SESSION = "session";
    static final String OFFSET = "offset";
    static final String LIMIT = "limit";
    static final String TIMEOUT = "timeout";
    static final String WAIT = "wait";
    static final String NOLOCK = "nolock";
    static final String FORMAT = "format";

    private static final String ROUTE_READ = "/pull/read?";
    private static final String ROUTE_COMMIT = "/pull/commit?";

    private static Field streamField;

    private final CloseableHttpClient client;
    private final RequestsStater stater;
    private final ClientConfig config;
    private final Partition partition;
    private final String id;
    private final PrefixedLogger logger;

    static {
        try {
            streamField =
                EofSensorInputStream.class.getDeclaredField("wrappedStream");
            AccessController.doPrivileged(
                new PrivilegedAction<Object>() {
                    @Override
                    public Object run() {
                        streamField.setAccessible(true);
                        return null;
                    }
                });
        } catch (NoSuchFieldException nfe) {
            throw new RuntimeException(nfe);
        }
    }

    // CSOFF: ParameterNumber
    public ChunkedSession(
        final ClientConfig config,
        final CloseableHttpClient client,
        final RequestsStater stater,
        final PrefixedLogger logger,
        final Partition partition,
        final String id)
    {
        this.client = client;
        this.stater = stater;
        this.config = config;
        this.partition = partition;
        this.id = id;
        this.logger = logger.replacePrefix(partition.idString());

        assert streamField != null;
    }
    // CSON: ParameterNumber

    @Override
    public String id() {
        return id;
    }

    @Override
    public Partition partition() {
        return partition;
    }

    @Override
    public void commit() throws LogbrockerClientException {
        commit(-1);
    }

    @Override
    public void commit(final long offset) throws LogbrockerClientException {
        QueryConstructor uri = new QueryConstructor(ROUTE_COMMIT);
        try {
            uri.append(TOPIC, partition.idString());
            uri.append(SESSION, id);
            uri.append(CLIENT, config.clientId());
            if (offset > 0) {
                uri.append(OFFSET, String.valueOf(offset));
            }
        } catch (BadRequestException bre) {
            throw new LogbrockerClientException(bre);
        }

        final long start = System.currentTimeMillis();
        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, uri.toString());
        logger.info(
            "Commit " + partition.host() + ' ' + request.toString());

        try (CloseableHttpResponse response =
                 client.execute(partition.host(), request))
        {
            int status = response.getStatusLine().getStatusCode();
            stater.accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    status,
                    start,
                    start,
                    0L,
                    AbstractLogbrokerClient.extractContentLength(response)));

            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            }

            String commitStatus = CharsetUtils.toString(response.getEntity());
            if (!commitStatus.trim().equalsIgnoreCase("ok")) {
                throw new BadResponseException(request, response);
            }
        } catch (IOException | HttpException e) {
            throw new LogbrockerClientException(e);
        }
    }

    @Override
    public ReadSession read() throws LogbrockerClientException {
        QueryConstructor uri = new QueryConstructor(ROUTE_READ);
        try {
            uri.append(CLIENT, config.clientId());
            uri.append(TOPIC, partition.idString());
            uri.append(FORMAT, config.format());
            uri.append(SESSION, id);

            if (config.limitRecords() != -1) {
                uri.append(LIMIT, String.valueOf(config.limitRecords()));
            }

            if (config.waitTimeout() != -1) {
                uri.append(TIMEOUT, String.valueOf(config.waitTimeout()));
            }

            if (config.waitData()) {
                uri.append(WAIT, String.valueOf(config.waitData()));
            }

            if (config.nolock()) {
                uri.append(NOLOCK, String.valueOf(config.nolock()));
            }
        } catch (BadRequestException bre) {
            throw new LogbrockerClientException(bre);
        }

        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, uri.toString());
        request.setHeader(
            HttpHeaders.ACCEPT_ENCODING,
            config.encoding());
        request.setHeader(
            HttpHeaders.ACCEPT_CHARSET,
            config.sessionHttpConfig().responseCharset().name());

        final long start = System.currentTimeMillis();
        CloseableHttpResponse response = null;
        ChunkedInputStream stream = null;
        InputStream originalStream = null;

        try {
            response = client.execute(partition.host(), request);
            int status = response.getStatusLine().getStatusCode();
            stater.accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    status,
                    start,
                    start,
                    0L,
                    AbstractLogbrokerClient.extractContentLength(response)));

            if (status != HttpStatus.SC_OK) {
                if (status == HttpStatus.SC_BAD_REQUEST) {
                    String body = EntityUtils.toString(response.getEntity());
                    if ("connections conflict".equalsIgnoreCase(body.trim())) {
                        throw new LogbrokerConnectionsConflictException(
                            request,
                            response,
                            body);
                    }
                }
                throw new BadResponseException(request, response);
            }

            originalStream = response.getEntity().getContent();
            stream = extractStream(originalStream);
        } catch (IOException | HttpException e) {
            try {
                if (response != null) {
                    response.close();
                }
            } catch (IOException ioe) {
                ioe.addSuppressed(e);
                throw new LogbrockerClientException(ioe);
            }

            throw new LogbrockerClientException(e);
        }

        return new ReadSession(stream, originalStream, this);
    }

    private static ChunkedInputStream extractStream(
        final InputStream stream)
        throws LogbrockerClientException
    {
        ChunkedInputStream result;

        try {
            result = (ChunkedInputStream) streamField.get(stream);
        } catch (IllegalAccessException iae) {
            throw new LogbrockerClientException(iae);
        }

        if (result == null) {
            throw new LogbrockerClientException(
                "Unsupported stream class " + stream.getClass().toString());
        }

        return result;
    }

    @Override
    public void close() throws IOException {
        client.close();
    }

    @Override
    public Charset charset() {
        return config.sessionHttpConfig().responseCharset();
    }

    @Override
    public String toString() {
        return "ChunkedSession{"
            + "partition=" + partition
            + ", id='" + id + '\''
            + '}';
    }
}
