package ru.yandex.logbroker.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.conn.BasicHttpClientConnectionManager;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.logbroker.client.exception.LogBrokerTransportException;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.client.exception.LogbrokerBadRequestException;
import ru.yandex.logbroker.client.exception.LogbrokerBadResponseException;
import ru.yandex.logbroker.client.exception.LogbrokerConflictException;
import ru.yandex.logbroker.client.exception.LogbrokerInvalidResponseException;
import ru.yandex.logbroker.client.exception.ParseException;
import ru.yandex.logbroker.config.ImmutableClientConfig;
import ru.yandex.logbroker.config.TopicClientConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.string.StringUtils;

public class DefaultLogbrokerClient
    extends SimpleLogbrokerClient
    implements AuthorizedLogbrokerClient
{
    // headers
    public static final String SESSION_HEADER = "Session";
    // request params
    // fields order in response of suggest request
    static final String[] SUGGEST_FIELDS = {"host:port", "topic:id"};

    // routes
    private static final String SUGGEST = "/pull/suggest?";
    private static final String SESSION = "/pull/session?";

    private final ImmutableClientConfig clientConfig;
    private final LogbrokerConnectionFactory connectionFactory;

    // CSOFF: ParameterNumber
    public DefaultLogbrokerClient(
        final TopicClientConfig topicConfig,
        final ImmutableDnsConfig dnsConfig,
        final PrefixedLogger logger,
        final RequestsStater stater)
    {
        super(topicConfig.hostConfig(), dnsConfig, stater, logger);

        clientConfig = topicConfig.clientConfig();
        connectionFactory = LogbrokerConnectionFactory.create(this.logger);
    }
    // CSON: ParameterNumber

    @Override
    public List<Partition> suggest(
        final List<String> topics,
        final String dc,
        final int count)
        throws LogbrockerClientException
    {
        List<Partition> result = new ArrayList<>();

        QueryConstructor uri = new QueryConstructor(SUGGEST);
        try {
            uri.append(CLIENT, clientConfig.clientId());
            if (topics.size() == 1) {
                uri.append(TOPIC, topics.get(0));
            } else {
                uri.append(TOPIC, StringUtils.join(topics, ','));
            }

            uri.append("count", String.valueOf(count));

            if (dc != null) {
                uri.append(DC, dc);
            }
        } catch (BadRequestException bre) {
            throw new LogbrockerClientException(bre);
        }

        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, uri.toString());
        logger.fine("Suggesting from " + hostConfig.host() + ' ' + request);

        try (CloseableHttpResponse response = execute(request)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                if (status == HttpStatus.SC_CONFLICT) {
                    throw new LogbrokerConflictException(request, response);
                }

                throw new BadResponseException(request, response);
            }

            String data = CharsetUtils.toString(response.getEntity());
            try {
                List<Map<String, String>> responseData =
                    LogbrokerClientResponseParser.parse(
                        SUGGEST_FIELDS,
                        data);

                for (Map<String, String> partition: responseData) {
                    result.add(BasicPartition.fromMap(partition));
                }
            } catch (ParseException pe) {
                throw new LogbrockerClientException(pe);
            }
        } catch (IOException | HttpException e) {
            throw new LogbrockerClientException(e);
        }

        return result;
    }

    @Override
    public List<Partition> suggest(final String topic, final int count)
        throws LogbrockerClientException
    {
        return suggest(Collections.singletonList(topic), DC_LOCAL, count);
    }

    @Override
    public List<Partition> suggest(final List<String> topics, final int count)
        throws LogbrockerClientException
    {
        return suggest(topics, DC_LOCAL, count);
    }

    @Override
    public List<OffsetInfo> offsets(
        final String topic)
        throws LogbrockerClientException
    {
        return this.offsets(clientConfig.clientId(), topic);
    }

    @Override
    public OffsetInfo offsets(final Partition partition)
        throws LogbrockerClientException
    {
        List<OffsetInfo> offsets =
            offsets(Collections.singletonList(partition));

        if (offsets.isEmpty()) {
            return null;
        }

        return offsets.get(0);
    }

    @Override
    public List<OffsetInfo> offsets(
        final List<Partition> partitions)
        throws LogbrockerClientException
    {
        if (partitions.size() == 0) {
            return Collections.emptyList();
        }

        String topic = partitions.get(0).topic();
        StringBuilder sb = new StringBuilder(topic);
        sb.append(":");
        for (int i = 0; i < partitions.size(); i++) {
            Partition partition = partitions.get(i);
            if (!partition.topic().equals(topic)) {
                throw new LogbrockerClientException(
                    "Requested partitions are from different topics");
            }

            sb.append(partition.id());
            if (i != partitions.size() - 1) {
                sb.append(',');
            }
        }

        List<Map<String, String>> offsets = offsetsList(sb.toString());

        List<OffsetInfo> result = new ArrayList<>();
        try {
            for (int i = 0; i < offsets.size(); i++) {
                result.add(
                    BasicOffsetInfo.fromMap(partitions.get(i), offsets.get(i)));
            }
        } catch (ParseException e) {
            throw new LogbrockerClientException(e);
        }

        if (offsets.size() != partitions.size()) {
            String message = "Logbroker returned not valid "
                + "offsets number " + offsets.toString();
            throw new LogbrockerClientException(message);
        }

        return result;
    }

    protected List<Map<String, String>> offsetsList(
        final String topic)
        throws LogbrockerClientException
    {
        return this.offsetsList(
            hostConfig.host(),
            clientConfig.clientId(),
            topic);
    }

    @Override
    public Session session(final Partition partition)
        throws LogbrockerClientException
    {
        return session(partition, false);
    }

    @Override
    public Session session(
        final Partition partition,
        final boolean transferable)
        throws LogbrockerClientException
    {
        QueryConstructor uri = new QueryConstructor(SESSION);
        try {
            uri.append(CLIENT, clientConfig.clientId());
            uri.append(TOPIC, partition.idString());
        } catch (BadRequestException bre) {
            throw new LogbrokerBadRequestException(bre);
        }

        CloseableHttpClient sessionClient = createSessionClient();
        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, uri.toString());
        if (clientConfig.sessionTimeout() > 0) {
            request.setHeader(
                "Timeout",
                String.valueOf(clientConfig.sessionTimeout()));
        }

        String sessionId;

        logger.fine(
            "Creating session to " + partition.host()
                + ' ' + request
                + ' ' + Arrays.toString(request.getAllHeaders()));

        try (CloseableHttpResponse response =
                 sessionClient.execute(partition.host(), request))
        {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                if (status == HttpStatus.SC_CONFLICT) {
                    throw new LogbrokerConflictException(request, response);
                }

                throw new LogbrokerBadResponseException(request, response);
            }

            Header sessionHeader = response.getFirstHeader(SESSION_HEADER);
            if (sessionHeader == null || sessionHeader.getValue().isEmpty()) {
                throw new LogbrokerInvalidResponseException(
                    "No session id was returned",
                    request,
                    response);
            }

            sessionId = sessionHeader.getValue();
        } catch (IOException | HttpException e) {
            throw new LogBrokerTransportException(e);
        }

        return new ChunkedSession(
            clientConfig,
            sessionClient,
            stater,
            logger,
            partition,
            sessionId);
    }

    private CloseableHttpClient createSessionClient() {
        BasicHttpClientConnectionManager connManager =
            new BasicHttpClientConnectionManager(
                RegistryBuilder.<ConnectionSocketFactory>create()
                    .register(
                        "http",
                        PlainConnectionSocketFactory.getSocketFactory())
                    .register(
                        "https",
                        SSLConnectionSocketFactory.getSocketFactory()).build(),
                connectionFactory,
                null,
                ClientBuilder.createDnsResolver(dnsConfig));

        connManager.setConnectionConfig(
            clientConfig.sessionHttpConfig().toConnectionConfig());

        connManager.setSocketConfig(
            ClientBuilder.createSocketConfig(clientConfig.sessionHttpConfig()));

        return ClientBuilder.createClient(
            clientConfig.sessionHttpConfig(),
            connManager);
    }

    @Override
    public String clientId() {
        return clientConfig.clientId();
    }
}
