package ru.yandex.logbroker.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;

import ru.yandex.http.util.CharsetUtils;
import ru.yandex.logbroker.client.exception.ParseException;

public final class LogbrokerClientResponseParser {
    public static final String FIELDS_HEADER = "Fields";
    public static final String TAB = "\t";
    public static final Pattern TAB_SEPARATOR = Pattern.compile(TAB);
    public static final String COLON = ":";
    public static final Pattern COLON_SEPARATOR = Pattern.compile(COLON);

    private static final String LINE_SEPARATOR = "\n";

    private LogbrokerClientResponseParser() {
    }

    public static List<Map<String, String>> parse(final HttpResponse response)
        throws ParseException, HttpException, IOException
    {
        Header fieldsHeader = response.getFirstHeader(FIELDS_HEADER);
        if (fieldsHeader == null) {
            throw new ParseException(
                FIELDS_HEADER,
                "Header was not found in response");
        }

        String data = CharsetUtils.toString(response.getEntity());
        return parse(fieldsHeader.getValue(), data);
    }

    public static List<Map<String, String>> parse(
        final String[] fieldsStr,
        final String data)
        throws ParseException
    {
        List<Map<String, String>> result = new ArrayList<>();

        Field[] fields = new Field[fieldsStr.length];
        for (int i = 0; i < fieldsStr.length; i++) {
            if (fieldsStr[i].contains(COLON)) {
                fields[i] = new MultiField(fieldsStr[i], COLON_SEPARATOR);
            } else {
                fields[i] = new SimpleField(fieldsStr[i]);
            }
        }

        String[] lines = data.split(LINE_SEPARATOR);
        for (String line: lines) {
            line = line.trim();
            if (line.isEmpty()) {
                continue;
            }

            Map<String, String> map = new HashMap<>();
            String[] values = TAB_SEPARATOR.split(line);
            if (values.length != fields.length) {
                throw new ParseException(
                    "Invalid format, declared fields "
                        + Arrays.toString(fields)
                        + " do not match values " + line);
            }

            for (int i = 0; i < fields.length; i++) {
                fields[i].parse(values[i]).addToMap(map);
            }

            result.add(map);
        }

        return result;
    }

    public static List<Map<String, String>> parse(
        final String fieldsStr,
        final String data)
        throws ParseException
    {
        if (data.isEmpty()) {
            return Collections.emptyList();
        }

        return parse(TAB_SEPARATOR.split(fieldsStr), data);
    }

    private interface Field {
        Field parse(final String value) throws ParseException;

        void addToMap(final Map<String, String> map);
    }

    private static final class MultiField implements Field {
        private final String[] names;
        private final Pattern separator;
        private String[] values;

        private MultiField(final String name, final Pattern separator) {
            this.separator = separator;
            this.names = this.separator.split(name);
        }

        @Override
        public Field parse(final String value) throws ParseException {
            this.values = this.separator.split(value);
            if (this.values.length != this.names.length) {
                throw new ParseException(
                    "Value parts do not equal fields size, fields: "
                        + Arrays.toString(names) + " value: " + value);
            }

            return this;
        }

        @Override
        public void addToMap(final Map<String, String> map) {
            for (int i = 0; i < names.length; i++) {
                map.put(names[i], values[i]);
            }
        }

        @Override
        public String toString() {
            return Arrays.toString(names);
        }
    }

    private static final class SimpleField implements Field {
        private final String name;
        private String value;

        private SimpleField(final String name) {
            this.name = name;
        }

        @Override
        public void addToMap(final Map<String, String> map) {
            map.put(name, value);
        }

        @Override
        public Field parse(final String value) {
            this.value = value;
            return this;
        }

        @Override
        public String toString() {
            return name;
        }
    }
}
