package ru.yandex.logbroker.client;

import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.conn.ManagedHttpClientConnection;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.entity.ContentLengthStrategy;
import org.apache.http.impl.conn.DefaultHttpResponseParserFactory;
import org.apache.http.impl.conn.ManagedHttpClientConnectionFactory;
import org.apache.http.impl.entity.LaxContentLengthStrategy;
import org.apache.http.impl.entity.StrictContentLengthStrategy;
import org.apache.http.impl.io.DefaultHttpRequestWriterFactory;
import org.apache.http.io.HttpMessageParserFactory;
import org.apache.http.io.HttpMessageWriterFactory;

import ru.yandex.logger.PrefixedLogger;

public class LogbrokerConnectionFactory
    extends ManagedHttpClientConnectionFactory
{
    public static final LogbrokerConnectionFactory INSTANCE =
        new LogbrokerConnectionFactory();

    private static final AtomicLong COUNTER = new AtomicLong();

    private final HttpMessageWriterFactory<HttpRequest> requestWriterFactory;
    private final HttpMessageParserFactory<HttpResponse> responseParserFactory;
    private final ContentLengthStrategy incomingContentStrategy;
    private final ContentLengthStrategy outgoingContentStrategy;
    private final PrefixedLogger logger;

    // CSOFF: ParameterNumber
    @SuppressWarnings("unchecked")
    public LogbrokerConnectionFactory(
        final HttpMessageWriterFactory<HttpRequest> requestWriterFactory,
        final HttpMessageParserFactory<HttpResponse> responseParserFactory,
        final ContentLengthStrategy incomingContentStrategy,
        final ContentLengthStrategy outgoingContentStrategy,
        final PrefixedLogger logger)
    {
        if (requestWriterFactory != null) {
            this.requestWriterFactory = requestWriterFactory;
        } else {
            this.requestWriterFactory =
                DefaultHttpRequestWriterFactory.INSTANCE;
        }

        if (responseParserFactory != null) {
            this.responseParserFactory = responseParserFactory;
        } else {
            this.responseParserFactory =
                DefaultHttpResponseParserFactory.INSTANCE;
        }

        if (incomingContentStrategy != null) {
            this.incomingContentStrategy = incomingContentStrategy;
        } else {
            this.incomingContentStrategy = LaxContentLengthStrategy.INSTANCE;
        }

        if (outgoingContentStrategy != null) {
            this.outgoingContentStrategy = outgoingContentStrategy;
        } else {
            this.outgoingContentStrategy = StrictContentLengthStrategy.INSTANCE;
        }

        if (logger != null) {
            this.logger = logger.addPrefix("ConnectionFactory");
        } else {
            this.logger = null;
        }
    }
    // CSON: ParameterNumber

    public LogbrokerConnectionFactory(
        final HttpMessageWriterFactory<HttpRequest> requestWriterFactory,
        final HttpMessageParserFactory<HttpResponse> responseParserFactory,
        final PrefixedLogger logger)
    {
        this(requestWriterFactory, responseParserFactory, null, null, logger);
    }

    public LogbrokerConnectionFactory(
        final HttpMessageParserFactory<HttpResponse> responseParserFactory,
        final PrefixedLogger logger)
    {
        this(null, responseParserFactory, logger);
    }

    public LogbrokerConnectionFactory(final PrefixedLogger logger) {
        this(null, null, logger);
    }

    public LogbrokerConnectionFactory() {
        this(null);
    }

    @Override
    public ManagedHttpClientConnection create(
        final HttpRoute route,
        final ConnectionConfig config)
    {
        ConnectionConfig cconfig = config;
        if (cconfig == null) {
            cconfig = ConnectionConfig.DEFAULT;
        }

        CharsetDecoder chardecoder = null;
        CharsetEncoder charencoder = null;
        Charset charset = cconfig.getCharset();
        CodingErrorAction malformedInputAction =
            cconfig.getMalformedInputAction();
        if (malformedInputAction == null) {
            malformedInputAction = CodingErrorAction.REPORT;
        }

        CodingErrorAction unmappableInputAction =
            cconfig.getUnmappableInputAction();
        if (unmappableInputAction == null) {
            unmappableInputAction = CodingErrorAction.REPORT;
        }

        if (charset != null) {
            chardecoder = charset.newDecoder();
            chardecoder.onMalformedInput(malformedInputAction);
            chardecoder.onUnmappableCharacter(unmappableInputAction);
            charencoder = charset.newEncoder();
            charencoder.onMalformedInput(malformedInputAction);
            charencoder.onUnmappableCharacter(unmappableInputAction);
        }

        String id = "http-outgoing-" + Long.toString(COUNTER.getAndIncrement());

        if (this.logger != null) {
            this.logger.info(
                "Creating session connection " + route.toString());
        }

        return new LogbrokerManagedHttpClientConnection(
            id,
            cconfig.getBufferSize(),
            cconfig.getFragmentSizeHint(),
            chardecoder,
            charencoder,
            cconfig.getMessageConstraints(),
            this.incomingContentStrategy,
            this.outgoingContentStrategy,
            this.requestWriterFactory,
            this.responseParserFactory,
            this.logger);
    }

    public static LogbrokerConnectionFactory create(
        final PrefixedLogger logger)
    {
        return new LogbrokerConnectionFactory(logger);
    }
}
