package ru.yandex.logbroker.client;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.RequestsStater;

public class MultidcLogbrokerClient extends AbstractLogbrokerClient {
    private final HttpHost host;

    // CSOFF: ParameterNumber
    public MultidcLogbrokerClient(
        final ImmutableHttpHostConfig config,
        final ImmutableDnsConfig dnsConfig,
        final RequestsStater stater,
        final PrefixedLogger logger)
    {
        super(config, dnsConfig, stater, logger);

        this.host = config.host();
    }
    // CSON: ParameterNumber

    public List<String> list(
        final String ident,
        final String logType)
        throws LogbrockerClientException
    {
        return super.list(host, ident, logType, false);
    }

    public Map<String, HttpHost> clusters() throws LogbrockerClientException {
        HttpRequest request =
            new BasicHttpRequest(HttpGet.METHOD_NAME, "/clusters");
        if (logger != null) {
            logger.fine("Cluster request from " + host + ' ' + request);
        }

        Map<String, HttpHost> result = new LinkedHashMap<>();
        try (CloseableHttpResponse response = execute(host, request)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            }

            String responseData = CharsetUtils.toString(response.getEntity());
            JsonObject json =
                TypesafeValueContentHandler.parse(responseData);

            for (Map.Entry<String, JsonObject> entry: json.asMap().entrySet()) {
                String balancerHost =
                    entry.getValue().asMap().get("balancer").asStringOrNull();
                if (balancerHost == null) {
                    throw new LogbrockerClientException(
                        "Bad balancer " + responseData);
                }
                result.put(
                    entry.getKey(),
                    new HttpHost(balancerHost, host.getPort()));
            }
        } catch (IOException | HttpException | JsonException e) {
            throw new LogbrockerClientException(e);
        }

        return result;
    }
}
