package ru.yandex.logbroker.config;

import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.PositiveIntegerValidator;

public class ClientConfigBuilder implements ClientConfig {
    private static final String COMMIT = "commit";
    private static final String SESSION_RETRY_DELAY = "session-retry-delay";
    private static final String NETWORK_RETRY_DELAY = "retry-delay-network";
    private static final String NOPARTITION_RETRY_DELAY =
        "retry-delay-nopartition";

    private static final String CLIENT_ID = "clientId";
    private static final String FORMAT = "format";
    private static final String ENCODING = "encoding";

    private String clientId;
    private String format;
    private String encoding;
    private HttpTargetConfigBuilder sessionHttpConfig;
    private int workers;
    private int commitInterval;
    private int limitRecords;
    private int waitTimeout;
    private long sessionTimeout;
    private int maximumSkipRecordThreshold;
    private long retryNoPartitionsDelay;
    private long sessionRetryDelay;
    private long networkRetryDelay;
    private long topicResolveDelay;
    private boolean commit;
    private boolean waitData;
    private boolean nolock;

    public ClientConfigBuilder() {
        this(ClientConfigDefaults.INSTANCE);
    }

    public ClientConfigBuilder(final IniConfig config) throws ConfigException {
        this(config, ClientConfigDefaults.INSTANCE);
    }

    public ClientConfigBuilder(final ClientConfig config) {
        this.clientId(config.clientId());
        this.format(config.format());
        this.encoding(config.encoding());
        this.sessionHttpConfig(config.sessionHttpConfig());
        this.workers(config.workers());
        this.commitInterval(config.commitInterval());
        this.sessionTimeout(config.sessionTimeout());
        this.maximumSkipRecordThreshold(config.maximumSkipRecordThreshold());
        this.retryNoPartitionsDelay(config.retryNoPartitionsDelay());
        this.commit(config.commit());
        this.nolock(config.nolock());
        this.waitData(config.waitData());
        this.limitRecords(config.limitRecords());
        this.waitTimeout(config.waitTimeout());
        this.sessionRetryDelay(config.sessionRetryDelay());
        this.networkRetryDelay(config.networkRetryDelay());
        topicResolveDelay(config.topicResolveDelay());
    }

    public ClientConfigBuilder(
        final IniConfig config,
        final ClientConfig defaults)
        throws ConfigException
    {
        this.clientId = config.getString(CLIENT_ID, defaults.clientId());
        this.format = config.getString(FORMAT, defaults.format());
        this.encoding = config.getString(ENCODING, defaults.encoding());
        this.sessionHttpConfig =
            new HttpTargetConfigBuilder(config, defaults.sessionHttpConfig());
        this.workers = config.getInt("workers", defaults.workers());
        this.commitInterval =
            config.getInt("commit-interval", defaults.commitInterval());
        this.sessionTimeout =
            config.getLong("session-timeout", defaults.sessionTimeout());

        this.maximumSkipRecordThreshold =
            config.getInt(
                "skip-threshold",
                defaults.maximumSkipRecordThreshold());

        this.sessionRetryDelay = config.getLong(
            SESSION_RETRY_DELAY,
            defaults.sessionRetryDelay());

        this.networkRetryDelay =
            config.getLong(NETWORK_RETRY_DELAY, defaults.networkRetryDelay());

        this.retryNoPartitionsDelay = config.getLong(
            NOPARTITION_RETRY_DELAY,
            defaults.retryNoPartitionsDelay());

        this.commit = config.getBoolean(COMMIT, defaults.commit());
        this.nolock = config.getBoolean("nolock", defaults.nolock());
        this.waitData = config.getBoolean("wait-data", defaults.waitData());
        this.limitRecords =
            config.get(
                "limit-records",
                defaults.limitRecords(),
                PositiveIntegerValidator.INSTANCE);

        this.waitTimeout =
            config.getInt("wait-timeout", defaults.waitTimeout());

        this.topicResolveDelay =
            config.getLong("topic-resolve-delay", defaults.topicResolveDelay());
    }

    @Override
    public String clientId() {
        return clientId;
    }

    public ClientConfigBuilder clientId(final String clientId) {
        this.clientId = clientId;
        return this;
    }

    @Override
    public String format() {
        return format;
    }

    public ClientConfigBuilder format(final String format) {
        this.format = format;
        return this;
    }

    @Override
    public String encoding() {
        return encoding;
    }

    public ClientConfigBuilder encoding(final String encoding) {
        this.encoding = encoding;
        return this;
    }

    @Override
    public HttpTargetConfigBuilder sessionHttpConfig() {
        return sessionHttpConfig;
    }

    public ClientConfigBuilder sessionHttpConfig(
        final HttpTargetConfig sessionHttpConfig)
    {
        this.sessionHttpConfig = new HttpTargetConfigBuilder(sessionHttpConfig);
        return this;
    }

    @Override
    public int workers() {
        return workers;
    }

    public ClientConfigBuilder workers(final int workers) {
        this.workers = workers;
        return this;
    }

    @Override
    public int commitInterval() {
        return commitInterval;
    }

    public ClientConfigBuilder commitInterval(final int commitInterval) {
        this.commitInterval = commitInterval;
        return this;
    }

    @Override
    public int limitRecords() {
        return limitRecords;
    }

    public ClientConfigBuilder limitRecords(final int limitRecords) {
        this.limitRecords = limitRecords;
        return this;
    }

    @Override
    public int waitTimeout() {
        return waitTimeout;
    }

    public ClientConfigBuilder waitTimeout(final int waitTimeout) {
        this.waitTimeout = waitTimeout;
        return this;
    }

    @Override
    public long sessionTimeout() {
        return sessionTimeout;
    }

    public ClientConfigBuilder sessionTimeout(final long sessionTimeout) {
        this.sessionTimeout = sessionTimeout;
        return this;
    }

    @Override
    public int maximumSkipRecordThreshold() {
        return maximumSkipRecordThreshold;
    }

    public ClientConfigBuilder maximumSkipRecordThreshold(
        final int maximumSkipRecordThreshold)
    {
        this.maximumSkipRecordThreshold = maximumSkipRecordThreshold;
        return this;
    }

    @Override
    public long retryNoPartitionsDelay() {
        return retryNoPartitionsDelay;
    }

    public ClientConfigBuilder retryNoPartitionsDelay(
        final long retryNoPartitionsDelay)
    {
        this.retryNoPartitionsDelay = retryNoPartitionsDelay;
        return this;
    }

    @Override
    public boolean commit() {
        return commit;
    }

    public ClientConfigBuilder commit(final boolean commit) {
        this.commit = commit;
        return this;
    }

    @Override
    public boolean waitData() {
        return waitData;
    }

    public ClientConfigBuilder waitData(final boolean waitData) {
        this.waitData = waitData;
        return this;
    }

    @Override
    public boolean nolock() {
        return nolock;
    }

    public ClientConfigBuilder nolock(final boolean nolock) {
        this.nolock = nolock;
        return this;
    }

    @Override
    public long sessionRetryDelay() {
        return sessionRetryDelay;
    }

    public ClientConfigBuilder sessionRetryDelay(final long sessionRetryDelay) {
        this.sessionRetryDelay = sessionRetryDelay;
        return this;
    }

    @Override
    public long networkRetryDelay() {
        return networkRetryDelay;
    }

    public ClientConfigBuilder networkRetryDelay(final long networkRetryDelay) {
        this.networkRetryDelay = networkRetryDelay;
        return this;
    }

    @Override
    public long topicResolveDelay() {
        return topicResolveDelay;
    }

    public ClientConfigBuilder topicResolveDelay(final long topicResolveDelay) {
        this.topicResolveDelay = topicResolveDelay;
        return this;
    }

    public ImmutableClientConfig build() throws ConfigException {
        return new ImmutableClientConfig(this);
    }
}
