package ru.yandex.logbroker.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.logbroker.log.consumer.LogConsumerFactoryType;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class LogConfigBuilder implements LogConfig {
    private ClientConfigBuilder clientConfig;
    private LogConsumerFactoryType consumerFactory;
    private IniConfig consumerFactorySection;
    private Map<String, List<String>> consumeMap;
    private int parserThreads;
    private int queueSize;
    private String topic;
    private String name;
    private String ident;
    private String logType;
    private boolean enabled;
    private long dcUpdatePeriod;
    private long metricsTimeFrame;

    public LogConfigBuilder() {
        this(LogConfigDefaults.INSTANCE);
    }

    public LogConfigBuilder(
        final String name,
        final IniConfig config)
        throws ConfigException
    {
        this(name, config, LogConfigDefaults.INSTANCE);
    }

    public LogConfigBuilder(final LogConfig config) {
        this.clientConfig(config.clientConfig());
        this.consumerFactory(config.consumerFactory());
        this.consumerFactorySection(config.consumerFactorySection());
        this.consumeMap(config.consumeMap());
        this.parserThreads(config.parserThreads());
        this.queueSize(config.queueSize());
        this.topic(config.topic());
        this.name(config.name());
        this.ident(config.ident());
        this.logType(config.logType());
        this.enabled(config.enabled());
        this.dcUpdatePeriod(config.dcUpdatePeriod());
        this.metricsTimeFrame(config.metricsTimeFrame());
    }

    public LogConfigBuilder(
        final String name,
        final IniConfig config,
        final LogConfig defaults)
        throws ConfigException
    {
        this.name = name;
        this.topic = config.getString("topic", null);
        this.logType = config.getString("log-type", null);
        this.ident = config.getString("ident", null);
        this.enabled = config.getBoolean("enabled", true);
        this.parserThreads =
            config.getInt("parser-threads", defaults.parserThreads());
        this.queueSize = config.getInt("queue-size", defaults.queueSize());

        consumeMap = new HashMap<>();
        IniConfig consumeMapSection = config.sectionOrNull("consumemap");
        if (consumeMapSection != null) {
            for (String key: consumeMapSection.keys()) {
                consumeMap.put(
                    key,
                    consumeMapSection.getAll(
                        key,
                        new CollectionParser<>(
                            NonEmptyValidator.TRIMMED,
                            ArrayList::new)));
            }
        }

        consumerFactorySection = config.sectionOrNull("consumer");
        if (consumerFactorySection == null) {
            consumerFactorySection = defaults.consumerFactorySection();
            this.consumerFactory = defaults.consumerFactory();
        } else {
            this.consumerFactory = consumerFactorySection.getEnum(
                LogConsumerFactoryType.class,
                "type");
        }

        this.clientConfig =
            new ClientConfigBuilder(config, defaults.clientConfig());

        this.dcUpdatePeriod =
            config.getLong("dc-update-interval", defaults.dcUpdatePeriod());

        metricsTimeFrame = config.get(
            "metrics-time-frame",
            defaults.metricsTimeFrame(),
            DurationParser.POSITIVE_LONG);
    }

    @Override
    public ClientConfigBuilder clientConfig() {
        return clientConfig;
    }

    public LogConfigBuilder clientConfig(
        final ClientConfig clientConfig)
    {
        this.clientConfig = new ClientConfigBuilder(clientConfig);
        return this;
    }

    @Override
    public LogConsumerFactoryType consumerFactory() {
        return consumerFactory;
    }

    public LogConfigBuilder consumerFactory(
        final LogConsumerFactoryType consumerFactory)
    {
        this.consumerFactory = consumerFactory;
        return this;
    }

    @Override
    public IniConfig consumerFactorySection() {
        return consumerFactorySection;
    }

    public LogConfigBuilder consumerFactorySection(
        final IniConfig consumerFactorySection)
    {
        this.consumerFactorySection = consumerFactorySection;
        return this;
    }

    @Override
    public Map<String, List<String>> consumeMap() {
        return consumeMap;
    }

    public LogConfigBuilder consumeMap(
        final Map<String, List<String>> consumeMap)
    {
        this.consumeMap = new HashMap<>(consumeMap);
        return this;
    }

    @Override
    public int parserThreads() {
        return parserThreads;
    }

    public LogConfigBuilder parserThreads(final int parserThreads) {
        this.parserThreads = parserThreads;
        return this;
    }

    @Override
    public int queueSize() {
        return queueSize;
    }

    public LogConfigBuilder queueSize(final int queueSize) {
        this.queueSize = queueSize;
        return this;
    }

    @Override
    public String topic() {
        return topic;
    }

    public LogConfigBuilder topic(final String topic) {
        this.topic = topic;
        return this;
    }

    @Override
    public String name() {
        return name;
    }

    public LogConfigBuilder name(final String name) {
        this.name = name;
        return this;
    }

    @Override
    public String ident() {
        return ident;
    }

    public LogConfigBuilder ident(final String ident) {
        this.ident = ident;
        return this;
    }

    @Override
    public String logType() {
        return logType;
    }

    public LogConfigBuilder logType(final String logType) {
        this.logType = logType;
        return this;
    }

    @Override
    public boolean enabled() {
        return enabled;
    }

    public LogConfigBuilder enabled(final boolean enabled) {
        this.enabled = enabled;
        return this;
    }

    @Override
    public long dcUpdatePeriod() {
        return dcUpdatePeriod;
    }

    public LogConfigBuilder dcUpdatePeriod(final long dcUpdatePeriod) {
        this.dcUpdatePeriod = dcUpdatePeriod;
        return this;
    }

    @Override
    public long metricsTimeFrame() {
        return metricsTimeFrame;
    }

    public LogConfigBuilder metricsTimeFrame(final long metricsTimeFrame) {
        this.metricsTimeFrame = metricsTimeFrame;
        return this;
    }
}
