package ru.yandex.logbroker.log;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.zip.GZIPInputStream;

import ru.yandex.json.parser.JsonException;
import ru.yandex.logbroker.log.consumer.ChunkProducer;
import ru.yandex.logbroker.log.consumer.LogConsumer;
import ru.yandex.logbroker.topic.DataChunk;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.tskv.TskvException;

public abstract class AbstractLogParser<T> implements LogParser {
    protected static final int POLL_TIMEOUT = 500;
    protected static final int WAIT_CONSUME_INTERVAL_CHECK = 10;
    protected static final int THOUSAND = 1000;

    protected final ChunkProducer chunkProducer;
    protected final LogConsumer<T> consumer;
    protected final PrefixedLogger logger;
    protected final ParseMetrics metrics;

    protected final boolean waitConsume;

    protected volatile String name = null;
    protected volatile boolean stop = false;
    protected volatile boolean stopped = false;

    protected long chunkStartParseTime = -1L;

    protected AbstractLogParser(
        final ChunkProducer chunkProducer,
        final LogConsumer<T> consumer,
        final ParseMetrics metrics,
        final PrefixedLogger logger)
    {
        this(chunkProducer, consumer, metrics, logger, true);
    }

    protected AbstractLogParser(
        final ChunkProducer chunkProducer,
        final LogConsumer<T> consumer,
        final ParseMetrics metrics,
        final PrefixedLogger logger,
        final boolean waitConsume)
    {
        this.chunkProducer = chunkProducer;
        this.consumer = consumer;
        this.metrics = metrics;
        this.logger = logger.addPrefix("LogParser");
        this.waitConsume = waitConsume;
    }

    @Override
    public boolean stopped() {
        return stopped;
    }

    @Override
    public void stop() {
        stop = true;
        logger.info("Stop requested");
    }

    @Override
    public synchronized long currentParseTime() {
        if (chunkStartParseTime > 0) {
            return System.currentTimeMillis() - chunkStartParseTime;
        }

        return -1L;
    }

    @Override
    public String name() {
        return name;
    }

    protected abstract long process(
        final Reader reader,
        final DataChunk chunk,
        final ChunkConsumeCallback callback)
        throws JsonException, TskvException, IOException;

    @Override
    public void run() {
        name = Thread.currentThread().getName();
        logger.info("Starting chunk consumer");

        boolean stoppedGracefully = false;
        while (true) {
            DataChunk chunk = null;

            try {
                chunk = chunkProducer.next(POLL_TIMEOUT);
                if (chunk == null) {
                    if (stop) {
                        logger.info("Gracefully stopping");
                        stoppedGracefully = true;
                        break;
                    }

                    continue;
                }

                long start = System.currentTimeMillis();
                synchronized (this) {
                    chunkStartParseTime = start;
                }

                InputStream stream;
                if ("gzip".equalsIgnoreCase(chunk.codec())) {
                    stream = new GZIPInputStream(chunk.data());
                } else if (chunk.codec() == null) {
                    stream = chunk.data();
                } else {
                    throw new IOException("Unsupported codec " + chunk.codec());
                }

                ChunkConsumeCallback callback;
                if (waitConsume) {
                    callback = new CountingChunkConsumeCallback();
                } else {
                    logger.info("This parser is NoWait");
                    callback = new NoWaitChunkConsumeCallback();
                }

                Reader reader = new InputStreamReader(stream, chunk.charset());

                metrics.logbrokerDelivery(
                    chunk.fetchTime() - chunk.createTime());

                consumer.batchStart(chunk);
                metrics.parseTime(process(reader, chunk, callback));
                consumer.batchEnd(callback);
                callback.sent();

                while (!stop && !callback.consumed()) {
                    Thread.sleep(WAIT_CONSUME_INTERVAL_CHECK);
                }

                if (stop && !callback.consumed()) {
                    logger.warning("Records still in air");
                } else {
                    metrics.consumeTime(System.currentTimeMillis() - start);
                }
            } catch (IOException
                | TskvException
                | JsonException
                | NumberFormatException ioe)
            {
                String message;
                if (chunk != null) {
                    message = "Failed parse chunk " + chunk.toShortInfo();
                } else {
                    message = "Chunk parse failed";
                }

                logger.log(Level.WARNING, message, ioe);

                metrics.parseError();
            } catch (InterruptedException ie) {
                logger.warning(
                    "LogParser interrupted");
                break;
            } finally {
                synchronized (this) {
                    chunkStartParseTime = -1L;
                }
            }
        }

        if (stoppedGracefully) {
            logger.info(
                "LogParser stopped gracefully, producer size: "
                    + chunkProducer.size());
        } else {
            logger.warning(
                "LogParser stopped ungracefully, unparsed chunks: "
                    + chunkProducer.size());
        }

        stopped = true;
    }

    private final class NoWaitChunkConsumeCallback
        implements ChunkConsumeCallback
    {
        @Override
        public boolean consumed() {
            return true;
        }

        @Override
        public void sent() {
        }

        @Override
        public void completed(final Object o) {
        }

        @Override
        public void failed(final Exception e) {
            metrics.consumeError();
        }

        @Override
        public void cancelled() {
            metrics.consumeError();
        }
    }

    private final class CountingChunkConsumeCallback
        implements ChunkConsumeCallback
    {
        private final AtomicInteger inAir;

        private CountingChunkConsumeCallback() {
            this.inAir = new AtomicInteger(0);
        }

        @Override
        public void sent() {
            inAir.incrementAndGet();
        }

        @Override
        public void completed(final Object o) {
            inAir.decrementAndGet();
        }

        @Override
        public void failed(final Exception e) {
            inAir.decrementAndGet();
            metrics.consumeError();
        }

        @Override
        public void cancelled() {
            inAir.decrementAndGet();
        }

        @Override
        public boolean consumed() {
            return inAir.get() <= 0;
        }
    }
}
