package ru.yandex.logbroker.log;

import java.io.IOException;
import java.io.Reader;

import ru.yandex.function.GenericConsumer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.logbroker.log.consumer.ChunkProducer;
import ru.yandex.logbroker.log.consumer.LogConsumer;
import ru.yandex.logbroker.topic.DataChunk;
import ru.yandex.logger.PrefixedLogger;

public class JsonLogParser extends AbstractLogParser<JsonMap> {
    //CSOFF: ParameterNumber
    public JsonLogParser(
        final ChunkProducer chunkProducer,
        final LogConsumer<JsonMap> consumer,
        final ParseMetrics metrics,
        final PrefixedLogger logger)
    {
        super(chunkProducer, consumer, metrics, logger);
    }

    public JsonLogParser(
        final ChunkProducer chunkProducer,
        final LogConsumer<JsonMap> consumer,
        final ParseMetrics metrics,
        final PrefixedLogger logger,
        final boolean waitConsume)
    {
        super(chunkProducer, consumer, metrics, logger, waitConsume);
    }
    //CSON: ParameterNumber

    @Override
    protected long process(
        final Reader reader,
        final DataChunk chunk,
        final ChunkConsumeCallback callback)
        throws JsonException, IOException
    {
        ParserResettingJsonConsumer jsonConsumer =
            new ParserResettingJsonConsumer(chunk, callback);

        JsonParser jsonParser = new JsonParser(
            new StackContentHandler(
                new TypesafeValueContentHandler(jsonConsumer)),
            true);
        jsonParser.parse(reader);
        return jsonConsumer.parseTime();
    }

    private final class ParserResettingJsonConsumer
        implements GenericConsumer<JsonObject, JsonException>
    {
        private final DataChunk chunk;
        private final ChunkConsumeCallback callback;
        private long parseTime = 0;
        private long lastTs = System.currentTimeMillis();

        private ParserResettingJsonConsumer(
            final DataChunk chunk,
            final ChunkConsumeCallback callback)
        {
            this.chunk = chunk;
            this.callback = callback;
        }

        @Override
        public void accept(final JsonObject recordObj) throws JsonException {
            JsonMap record = recordObj.asMap();
            long unixtime = record.getLong("unixtime_ms");

            parseTime += System.currentTimeMillis() - lastTs;

            metrics.recordParsed();
            metrics.logbrokerWaitlog(chunk.createTime() - unixtime);
            callback.sent();
            consumer.apply(record, callback);
            metrics.timeLag(
                System.currentTimeMillis() - unixtime);
            lastTs = System.currentTimeMillis();
        }

        public long parseTime() {
            return parseTime;
        }
    }
}
