package ru.yandex.logbroker.log;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.logbroker.LogbrokerDc;
import ru.yandex.logbroker.Stoppable;
import ru.yandex.logbroker.client.DefaultLogbrokerClient;
import ru.yandex.logbroker.client.MultidcLogbrokerClient;
import ru.yandex.logbroker.client.SimpleLogbrokerClient;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.config.ImmutableLogConfig;
import ru.yandex.logbroker.config.ImmutableLogbrokerConsumerServerConfig;
import ru.yandex.logbroker.config.TopicClientConfig;
import ru.yandex.logbroker.config.TopicConfigBuilder;
import ru.yandex.logbroker.log.consumer.ChunksQueue;
import ru.yandex.logbroker.log.consumer.LogConsumerFactory;
import ru.yandex.logbroker.topic.TopicConsumeManager;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.StaterConfigBuilder;
import ru.yandex.stater.StatsConsumer;

public class LogConsumeManager
    extends AbstractStatable
    implements GenericAutoCloseable<IOException>, Stoppable
{
    private static final String LG_CLIENT_STATER = "/lbclient_";

    private final String currentDc;
    private final ThreadGroup threadGroup;
    private final ImmutableLogConfig logConfig;
    private final ImmutableLogbrokerConsumerServerConfig serverConfig;
    private final PrefixedLogger logger;
    private final ChunksQueue chunksQueue;
    private final List<TopicConsumeManager> consumeManagers;
    private final LogLagsStat logLagsStat;
    private final LogConsumerFactory consumerFactory;
    private final List<LogParser> parsers;
    private final ParseMetrics parseMetrics;
    private final ImmutableHttpHostConfig rootLogbrokerClientConfig;

    private final ExecutorService executor;
    private final GenericAutoCloseableChain<IOException> closeChain;
    private volatile boolean stop = false;
    private volatile boolean stopped = true;

    // CSOFF: ParameterNumber
    public LogConsumeManager(
        final ImmutableLogConfig logConfig,
        final SharedConnectingIOReactor reactor,
        final ImmutableLogbrokerConsumerServerConfig serverConfig,
        final PrefixedLogger logger,
        final String currentDc)
        throws ConfigException
    {
        this.logConfig = logConfig;
        this.serverConfig = serverConfig;
        this.currentDc = currentDc;
        this.threadGroup = new ThreadGroup(logConfig.name());
        this.logger = logger.addPrefix(logConfig.name());
        this.rootLogbrokerClientConfig =
            serverConfig.consumerConfig().logbrokerHostConfig();

        this.closeChain = new GenericAutoCloseableChain<>();

        this.chunksQueue = new ChunksQueue(logger, logConfig.queueSize());
        this.executor =
            Executors.newFixedThreadPool(
                logConfig.parserThreads(),
                new NamedThreadFactory(
                    new ThreadFactoryConfig("LogParser-")
                        .group(threadGroup)
                        .uncaughtExceptionHandler(
                            (t, e) -> {
                                logger.log(
                                    Level.SEVERE,
                                    "Log parser thread uncaught exception",
                                    e);
                            })));
        this.parseMetrics =
            new ParseMetrics(logConfig.name(), serverConfig.metricsTimeFrame());

        this.consumeManagers = new ArrayList<>();
        this.parsers = new ArrayList<>();

        this.logLagsStat =
            new LogLagsStat(
                logConfig,
                new MultidcLogbrokerClient(
                    rootLogbrokerClientConfig,
                    serverConfig.dnsConfig(),
                    clientStater(),
                    logger),
                logger);

        closeChain.add(logLagsStat);

        this.consumerFactory = logConfig.consumerFactory().create(
            reactor,
            serverConfig,
            logConfig.consumerFactorySection());

        closeChain.add(this.consumerFactory);

        this.stopped = true;
    }
    // CSON: ParameterNumber

    public String dc() {
        return currentDc;
    }

    public synchronized void addDc(final LogbrokerDc dc) {
        logger.info("Got new dc " + dc.toString() + " trying to add");
        logLagsStat.addDc(dc);

        if (logConfig.topic() == null) {
            List<String> topicsDcs = logConfig.consumeMap().get(currentDc);
            if (topicsDcs == null) {
                topicsDcs = Collections.singletonList(currentDc);
            }

            if (topicsDcs.contains(dc.name())) {
                try {
                    TopicConsumeManager manager =
                        createTopicConsumer(rootLogbrokerClientConfig, dc);

                    consumeManagers.add(manager);
                    //closeChain.add(manager);

                    manager.start();
                } catch (ConfigException ce) {
                    logger.log(
                        Level.WARNING,
                        "Unable to add dc " + dc.name(),
                        ce);
                }
            } else {
                logger.warning("New dc found, but it is not in consumemap");
            }
        }
    }

    public void removeDc(
        final LogbrokerDc dc,
        final long stopDeadline)
    {
        logger.warning("Dc removed " + dc.toString());
        logLagsStat.removeDc(dc);
        List<TopicConsumeManager> managersToStop = new ArrayList<>();
        synchronized (this) {
            Iterator<TopicConsumeManager> iterator = consumeManagers.iterator();
            while (iterator.hasNext()) {
                TopicConsumeManager manager = iterator.next();
                if (manager.dc().equalsIgnoreCase(dc.name())) {
                    managersToStop.add(manager);
                    iterator.remove();
                }
            }
        }

        try {
            for (TopicConsumeManager manager: managersToStop) {
                manager.waitStop(stopDeadline);
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Stopping managers was interrupted", ie);
        }
    }

    public LogConsumerFactory consumerFactory() {
        return consumerFactory;
    }

    public synchronized void start() {
        this.logLagsStat.start();
        consumeManagers.forEach(TopicConsumeManager::start);

        for (int i = 0; i < logConfig.parserThreads(); i++) {
            LogParser parser =
                consumerFactory.create(
                    logConfig,
                    chunksQueue,
                    parseMetrics,
                    logger);

            this.parsers.add(parser);
            this.executor.execute(parser);
        }
    }

    @Override
    public void close() throws IOException {
        this.executor.shutdownNow();

        IOException suppressed = null;
        for (TopicConsumeManager manager: consumeManagers) {
            try {
                manager.close();
            } catch (IOException ioe) {
                if (suppressed != null) {
                    ioe.addSuppressed(suppressed);
                }

                suppressed = ioe;
            }
        }

        closeChain.close();
        if (suppressed != null) {
            throw suppressed;
        }
    }

    @Override
    public synchronized void stop() {
        this.executor.shutdown();
        this.stop = true;

        this.consumeManagers.forEach(TopicConsumeManager::stop);
        this.parsers.forEach(Stoppable::stop);
        this.logLagsStat.stop();
    }

    @Override
    public synchronized boolean stopped() {
        return stopped;
    }

    @Override
    public boolean waitStop(final long deadline) throws InterruptedException {
        boolean status = true;
        for (Stoppable manager: this.consumeManagers) {
            status &= manager.waitStop(deadline);
        }

        if (!status) {
            logger.warning("Failed to stop consume managers");
        }

        for (Stoppable parser: parsers) {
            status &= parser.waitStop(deadline);
        }

        status &= this.logLagsStat.waitStop(deadline);
        if (!status) {
            logger.warning("Log stop failed");
        }

        return status;
    }

    @Override
    public synchronized <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        super.stats(statsConsumer);

        for (TopicConsumeManager consumeManager: consumeManagers) {
            consumeManager.stats(statsConsumer);
        }
        consumerFactory.stats(statsConsumer);
        parseMetrics.stats(statsConsumer);
        logLagsStat.stats(statsConsumer);
    }

    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> result = new LinkedHashMap<>();
        result.put(consumerFactory.name(), consumerFactory.status(verbose));
        result.put("queue-size", chunksQueue.size());
        for (int index = 0; index < parsers.size(); index++) {
            LogParser parser = parsers.get(index);
            result.put(
                parser.name(),
                parser.currentParseTime());
        }
        return result;
    }

    private RequestsStater clientStater() {
        return
            new RequestsStater(
                new StaterConfigBuilder()
                    .prefix(LG_CLIENT_STATER + logConfig.name()).build());
    }

    private DefaultLogbrokerClient client(
        final TopicClientConfig config)
    {
        return new DefaultLogbrokerClient(
            config,
            serverConfig.dnsConfig(),
            logger,
            clientStater());
    }

    private String topic(final SimpleLogbrokerClient lbClient) {
        String topic = null;
        try {
            while (!stop) {
                try {
                    List<String> topics =
                        lbClient.list(
                            logConfig.ident(),
                            logConfig.logType());

                    if (topics.size() == 1) {
                        topic = topics.get(0);
                        break;
                    }

                    logger.warning(
                        "Invalid topics count got " + topics.size());
                } catch (LogbrockerClientException lbe) {
                    logger.log(
                        Level.WARNING,
                        "Failed to get topics list",
                        lbe);
                }

                Thread.sleep(logConfig.clientConfig().topicResolveDelay());
            }
        } catch (InterruptedException ie) {
            logger.log(Level.SEVERE, "Failed retrieve topic", ie);
        }

        return topic;
    }

    // CSOFF: ParameterNumber
    private TopicConsumeManager createTopicConsumer(
        final HttpHostConfig clientConfig,
        final LogbrokerDc dc)
        throws ConfigException
    {
        return this.createTopicConsumer(clientConfig, dc, null);
    }

    private TopicConsumeManager createTopicConsumer(
        final HttpHostConfig clientConfig,
        final LogbrokerDc dc,
        final String topicPropose)
        throws ConfigException
    {
        //first we creating client for topic
        HttpHostConfigBuilder hostBuilder =
            new HttpHostConfigBuilder(clientConfig);
        hostBuilder.host(dc.balancer());

        TopicClientConfig topicClientConfig =
            new TopicClientConfig(
                hostBuilder.build(),
                logConfig.clientConfig());

        DefaultLogbrokerClient client =
            client(topicClientConfig);

        String topic = topicPropose;
        if (topic == null) {
            topic = topic(client);
            if (topic == null) {
                throw new ConfigException(
                    "Failed to resolve topic for "
                        + logConfig.name()
                        + " dc " + dc);
            }
        }

        TopicConfigBuilder topicConfig =
            new TopicConfigBuilder()
                .logName(logConfig.name())
                .consumer(chunksQueue)
                .config(topicClientConfig)
                .dc(dc.name())
                .topic(topic)
                .logger(logger)
                .client(client);

        return new TopicConsumeManager(
            topicConfig.build(),
            serverConfig.metricsTimeFrame());
    }
    // CSON: ParameterNumber
}
