package ru.yandex.logbroker.log;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logbroker.DcUpdateListener;
import ru.yandex.logbroker.LogbrokerDc;
import ru.yandex.logbroker.Stoppable;
import ru.yandex.logbroker.client.MultidcLogbrokerClient;
import ru.yandex.logbroker.client.OffsetInfo;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.config.ImmutableLogConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class LogLagsStat
    extends AbstractStatable
    implements Runnable, Stoppable, DcUpdateListener,
    GenericAutoCloseable<IOException>
{
    private static final char SEP = '-';
    private static final String THREAD_NAME = "LogStat";
    private static final String TOTAL_LAG = "-total-lag_axxx";
    private static final String MAX_LAG = "-max-lag_axxx";

    private final PrefixedLogger logger;
    private final ImmutableLogConfig logConfig;
    private final MultidcLogbrokerClient client;
    private final String clientId;

    private Map<LogbrokerDc, DcLagStater> dcLags =
        new LinkedHashMap<>();
    private final Thread thread;
    private final UnitedLagStater totalStater;
    private long lastTopicUpdate = 0;
    private volatile boolean stop = false;
    private volatile boolean stopped = false;

    public LogLagsStat(
        final ImmutableLogConfig logConfig,
        final MultidcLogbrokerClient client,
        final PrefixedLogger logger)
    {
        this.logConfig = logConfig;

        this.logger = logger.addPrefix(THREAD_NAME + logConfig.name());
        this.client = client;
        this.clientId = logConfig.clientConfig().clientId();
        this.totalStater = new UnitedLagStater(
            new NamedStatsAggregatorFactory<>(
                logConfig.name() + MAX_LAG,
                new MaxAggregatorFactory(0)));

        this.thread = new Thread(new ThreadGroup(THREAD_NAME), this);
    }

    public void start() {
        this.thread.start();
    }

    @Override
    public synchronized void removeDc(final LogbrokerDc dc) {
        this.dcLags.remove(dc);
    }

    @Override
    public synchronized void addDc(final LogbrokerDc dc) {
        this.dcLags.put(dc, stater(dc.name()));
    }

    public void updateTopics(final long now) throws LogbrockerClientException {
        if (now - lastTopicUpdate < logConfig.metricsTimeFrame() * 2) {
            return;
        }

        logger.info("Updating topics");
        lastTopicUpdate = now;

        Set<LogbrokerDc> dcList;
        synchronized (this) {
            dcList = dcLags.keySet();
        }

        Map<LogbrokerDc, List<String>> topicMap = new LinkedHashMap<>();
        for (LogbrokerDc dc: dcList) {
            topicMap.put(
                dc,
                client.list(
                    dc.balancer(),
                    logConfig.ident(),
                    logConfig.logType()));
        }

        synchronized (this) {
            for (Map.Entry<LogbrokerDc, List<String>> dc: topicMap.entrySet()) {
                dcLags.computeIfPresent(
                    dc.getKey(),
                    (k, v) -> v.topics(dc.getValue()));
            }
        }
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Collection<DcLagStater> staters;
        synchronized (this) {
            staters = dcLags.values();
        }

        for (DcLagStater stater: staters) {
            stater.stater().stats(statsConsumer);
        }

        totalStater.stats(statsConsumer);
    }

    private DcLagStater stater(final String dc) {
        TimeFrameQueue<Long> frameQueue =
            new TimeFrameQueue<>(logConfig.metricsTimeFrame());

        totalStater.add(dc, frameQueue);
        Stater stater =
            new PassiveStaterAdapter<>(
                frameQueue,
                new NamedStatsAggregatorFactory<>(
                    logConfig.name() + SEP + dc + TOTAL_LAG,
                    new MaxAggregatorFactory(0)));

        return new DcLagStater(frameQueue, stater);
    }

    @Override
    public void stop() {
        stop = true;
        this.thread.interrupt();
    }

    @Override
    public boolean stopped() {
        return stopped;
    }

    @Override
    public void close() throws IOException {
        this.thread.interrupt();
    }

    @Override
    public void run() {
        PrefixedLogger logger = this.logger.addPrefix(THREAD_NAME);

        try {
            while (!stop) {
                long now = System.currentTimeMillis();

                try {
                    updateTopics(now);
                } catch (LogbrockerClientException lbe) {
                    logger.log(Level.WARNING, "Failed to update topics", lbe);
                }

                Map<LogbrokerDc, DcLagStater> local;
                synchronized (this) {
                    local = new LinkedHashMap<>(dcLags);
                }

                for (Map.Entry<LogbrokerDc, DcLagStater> entry
                    : local.entrySet())
                {
                    LogbrokerDc dc = entry.getKey();

                    long sum = 0;
                    for (String topic: entry.getValue().topics()) {
                        try {
                            List<OffsetInfo> offsets =
                                client.offsets(
                                    dc.balancer(),
                                    clientId,
                                    topic);
                            for (OffsetInfo info: offsets) {
                                sum += info.lag();
                            }
                        } catch (LogbrockerClientException lbe) {
                            logger.log(
                                Level.WARNING,
                                "Failed to retrieve offsets for " + topic);
                        }
                    }

                    entry.getValue().frameQueue().accept(sum);
                }

                Thread.sleep(logConfig.metricsTimeFrame() / 2);
            }
        } catch (InterruptedException ie) {
            logger.info("Stat thread interrupted");
        }

        this.stopped = true;
    }

    private static final class DcLagStater {
        private final TimeFrameQueue<Long> frameQueue;
        private final Stater stater;

        private List<String> topics;

        private DcLagStater(
            final TimeFrameQueue<Long> frameQueue,
            final Stater stater)
        {
            this.frameQueue = frameQueue;
            this.stater = stater;
            this.topics = new ArrayList<>();
        }

        public DcLagStater topics(final List<String> topics) {
            this.topics = topics;
            return this;
        }

        public List<String> topics() {
            return topics;
        }

        public TimeFrameQueue<Long> frameQueue() {
            return frameQueue;
        }

        public Stater stater() {
            return stater;
        }
    }
}
