package ru.yandex.logbroker.log.consumer;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import ru.yandex.logbroker.topic.ChunkConsumer;
import ru.yandex.logbroker.topic.DataChunk;
import ru.yandex.logger.PrefixedLogger;

public class ChunksQueue implements ChunkConsumer, ChunkProducer {
    private final LinkedBlockingQueue<DataChunk> queue;
    private final PrefixedLogger logger;
    private final Map<String, Long> consumeOffsets = new ConcurrentHashMap<>();
    private final Map<String, Long> addOffsets = new ConcurrentHashMap<>();

    public ChunksQueue(final PrefixedLogger logger, final int size) {
        this.queue = new LinkedBlockingQueue<>(size);
        this.logger = logger;
    }

    @Override
    public boolean chunk(final DataChunk chunk) {
        Long offset = addOffsets.get(chunk.partition());
        if (offset != null && chunk.offset() <= offset) {
            this.logger.warning(
                "Skipping stale chunk add, lastOffset is " + offset
                    + " adding is " + chunk.offset()
                    + " parition is " + chunk.partition());
            return true;
        }

        this.addOffsets.put(chunk.partition(), chunk.offset());

        boolean status = true;
        try {
            this.queue.put(chunk);
        } catch (InterruptedException ie) {
            status = false;
        }

        return status;
    }

    @Override
    public synchronized DataChunk next(final long timeout)
        throws InterruptedException
    {
        DataChunk chunk = this.queue.poll(timeout, TimeUnit.MILLISECONDS);
        if (chunk == null) {
            return null;
        }

        consumeOffsets.put(chunk.partition(), chunk.offset());
        return chunk;
    }

    @Override
    public synchronized DataChunk next() throws InterruptedException {
        DataChunk chunk = this.queue.take();
        consumeOffsets.put(chunk.partition(), chunk.offset());
        return chunk;
    }

    @Override
    public long lastOffset(final String topic) {
        Long offset = consumeOffsets.get(topic);
        if (offset == null) {
            return -1;
        }

        return offset;
    }

    @Override
    public int size() {
        return queue.size();
    }
}
