package ru.yandex.logbroker.log.consumer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.config.URIConfigBuilder;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logbroker.config.ImmutableLogbrokerConsumerServerConfig;
import ru.yandex.logbroker.config.LogConfig;
import ru.yandex.logbroker.log.LogbrokerTskvRecord;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.tskv.TskvRecord;

public class HttpPostLogConsumerFactory implements TskvLogConsumerFactory {
    public static final String NAME = "post";

    private final AsyncClient client;
    private final ImmutableURIConfig config;
    private final Collection<String> fields;
    private final boolean skipOnMising;

    public HttpPostLogConsumerFactory(
        final SharedConnectingIOReactor reactor,
        final ImmutableLogbrokerConsumerServerConfig serverConfig,
        final IniConfig config)
        throws ConfigException
    {
        this.config = new URIConfigBuilder(config).build();
        this.skipOnMising = config.getBoolean("skip-missing", false);

        fields =
            config.getAll(
                "fields",
                Collections.emptyList(),
                new CollectionParser<>(String::trim, ArrayList::new));

        client =
            new AsyncClient(reactor, this.config).adjustStater(
                serverConfig.staters(),
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/post/client")));

        client.start();
    }

    @Override
    public LogConsumer<LogbrokerTskvRecord> create(
        final LogConfig logConfig,
        final PrefixedLogger logger)
    {
        return new HttpLogConsumer();
    }

    private final class HttpLogConsumer
        implements LogConsumer<LogbrokerTskvRecord>
    {
        private void withFields(
            final JsonWriter writer,
            final TskvRecord record)
            throws IOException
        {
            writer.startObject();
            for (String field : fields) {
                String value = record.get(field);
                if (value != null || !skipOnMising) {
                    writer.key(field);
                    writer.value(value);
                }
            }

            writer.endObject();
        }

        @Override
        public void apply(
            final LogbrokerTskvRecord record,
            final FutureCallback<Object> callback)
        {
            StringBuilderWriter sbWriter = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(sbWriter)) {
                if (fields.isEmpty()) {
                    writer.value(record);
                } else {
                    withFields(writer, record);
                }
            } catch (IOException ioe) {
                callback.failed(ioe);
                return;
            }

            client.execute(
                config.host(),
                new BasicAsyncRequestProducerGenerator(
                    config.uri().toString(),
                    sbWriter.toString()),
                JsonAsyncDomConsumerFactory.OK,
                callback);
        }
    }

    @Override
    public void close() throws IOException {
        client.close();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("post-client", client.status(verbose));
        return status;
    }
}
