package ru.yandex.logbroker.log.consumer.blackbox;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import ru.yandex.tskv.InplaceTskvParser;
import ru.yandex.tskv.TskvException;
import ru.yandex.tskv.TskvFormatConfigBuilder;
import ru.yandex.tskv.TskvRecord;

public class BlackboxEventsParser {
    private static final String UID_SEPARATOR = ",";
    private static final String MAIL_MOBILE = "mobmail:all";

    private final InplaceTskvParser tskvParser;
    private final BlackboxConsumerContext context;

    public BlackboxEventsParser(final BlackboxConsumerContext context) {
        this.context = context;
        this.tskvParser =
            new InplaceTskvParser(
                new TskvFormatConfigBuilder().fieldSeparator(';'));
    }

    private SessionCheck parse(
        final TskvRecord record,
        final String uidField,
        final String host)
    {
        String uidsStr = record.get(uidField);
        String unixtimeStr = record.get("unixtime");
        if (uidsStr == null || unixtimeStr == null) {
            context.logger().warning(
                "No uids or unixtime " + record.toString());
            context.parseError();
            return null;
        }

        String userIp = record.get("user_ip");
        if (userIp == null) {
            userIp = record.getOrDefault("userip", "");
        }

        String yuid = record.getOrNull("yuid");

        long unixtime;
        List<Long> uids;

        SessionCheck result = null;
        try {
            uidsStr = uidsStr.trim();

            String[] split = uidsStr.split(UID_SEPARATOR);
            uids = new ArrayList<>(split.length);
            for (String s: split) {
                s = s.trim();
                if (!s.isEmpty()) {
                    uids.add(Long.parseLong(s));
                }
            }

            unixtime = Long.parseLong(unixtimeStr);

            result = new SessionCheck(uids, host, yuid, userIp, unixtime);
        } catch (NumberFormatException nfe) {
            String message =
                "BadFormat uid or unixtime " + uidsStr + ' ' + unixtimeStr;
            context.logger().warning(message);
            context.parseError();
        }

        return result;
    }

    // CSOFF: ReturnCount
    public SessionCheck parse(final TskvRecord record) {
        String action = record.get("action");

        if ("sescheck".equalsIgnoreCase(action)) {
            String host = record.get("host");
            if (host == null) {
                host = "";
            }
//            if (!host.toLowerCase(Locale.ROOT).startsWith("mail.yandex.")) {
//                return null;
//            }

            return parse(record, "uids", host);
        } else if ("auth".equalsIgnoreCase(action)) {
            String commentStr = record.get("comment");
            if (commentStr == null || commentStr.isEmpty()) {
                return null;
            }

            String type = record.get("type");
            if (!"oauthcheck".equalsIgnoreCase(type)) {
                return null;
            }

            String status = record.get("status");
            if ("failed".equalsIgnoreCase(status)) {
                // scope for failed status always null
                return null;
            }

            TskvRecord comment = null;
            try {
                if (commentStr.endsWith(";")) {
                    commentStr =
                        commentStr.substring(0, commentStr.length() - 1);
                }

                if (!commentStr.contains("=")) {
                    return null;
                }

                comment = this.tskvParser.next(new StringReader(commentStr));
            } catch (IOException | TskvException e) {
                context.logger().log(
                    Level.WARNING,
                    "Auth Parse error for " + record.toString(),
                    e);
                context.parseError();
            }

            if (comment == null) {
                return null;
            }

            String scope = comment.get("scope");
            if (scope == null || scope.isEmpty()) {
                // scope null is ok, for example for failed auth
                return null;
            }

            if (!scope.contains(MAIL_MOBILE)) {
                return null;
            }

            String devnm = comment.getOrDefault("devnm", "");
            return parse(record, "uid", "mobile_" + devnm);
        }

        return null;
    }
    // CSON: ReturnCount
}
