package ru.yandex.logbroker.log.consumer.blackbox;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.logbroker.log.LogbrokerTskvRecord;
import ru.yandex.logbroker.log.consumer.LogConsumer;
import ru.yandex.logbroker.topic.DataChunk;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.SearchMapShard;

public class BlackboxLogConsumer implements LogConsumer<LogbrokerTskvRecord> {
    private static final int MILLS_IN_SEC = 1000;
    private static final int DEFAULT_USERS_SIZE = 10;
    private static final int DEFAULT_MULTIPLIER = 10;
    private static final int DEFAULT_FLUSH_INTERVAL = 200;
    private static final int MAX_DELAY = 1000000;

    private static final String URI = "/update?&service=major&prefix=";

    private final BlackboxConsumerContext context;
    private final BlackboxEventsParser parser;

    private Map<SearchMapShard, Map<Long, AggregatedUser>> writersMap;
    private long lastFlush = 0;
    private boolean skip = false;

    public BlackboxLogConsumer(
        final BlackboxConsumerContext context,
        final PrefixedLogger logger)
    {
        this.context = context;
        this.parser = new BlackboxEventsParser(context);
        this.writersMap = new HashMap<>();
    }

    @Override
    public void batchStart(final DataChunk chunk) {
        if (System.currentTimeMillis() - chunk.createTime() > MAX_DELAY) {
            skip = true;
        } else {
            skip = false;
        }
    }

    @Override
    public void batchEnd(
        final FutureCallback<Object> callback)
    {
        long cts = System.currentTimeMillis();
        if (skip || cts - lastFlush < DEFAULT_FLUSH_INTERVAL) {
            callback.completed(null);
            return;
        }

        lastFlush = cts;
        //context.logger().info(writersMapToString(writersMap.keySet()));
        MultiFutureCallback<Void> mfcb =
            new MultiFutureCallback<>(callback);

        for (Map.Entry<SearchMapShard, Map<Long, AggregatedUser>> entry
            : writersMap.entrySet())
        {
            StringBuilderWriter sbw =
                new StringBuilderWriter(
                    new StringBuilder(
                        DEFAULT_MULTIPLIER * entry.getValue().size()));

            Long prefix = entry.getValue().keySet().iterator().next();
            if (prefix == null) {
                context.logger().warning("Null prefix " + writersMap);
                context.parseError();
                continue;
            }

            FutureCallback<Void> hostCallback =
                mfcb.newCallback();

            try (JsonWriter writer = new JsonWriter(sbw)) {
                writer.startObject();
                for (Map.Entry<Long, AggregatedUser> user
                    : entry.getValue().entrySet())
                {
                    prefix = user.getKey();
                    writer.key(user.getKey().toString());
                    writer.value(user.getValue());
                }
                writer.endObject();
            } catch (IOException ioe) {
                hostCallback.failed(ioe);
                continue;
            }

            List<HttpHost> hosts = new ArrayList<>(entry.getKey().size());
            entry.getKey().forEach((s) -> hosts.add(s.indexerHost()));

            if (hosts.size() <= 0) {
                context.logger().warning("No hosts for " + prefix);
                context.parseError();
                hostCallback.failed(
                    new Exception("Bad searchmap for " + prefix));
                continue;
            }
            context.client().execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(
                    URI + prefix, sbw.toString()),
                EmptyAsyncConsumerFactory.ANY_GOOD,
                hostCallback);
        }

        writersMap.clear();
        mfcb.done();
    }

    @Override
    public void apply(
        final LogbrokerTskvRecord record,
        final FutureCallback<Object> callback)
    {
        callback.completed(null);
        if (skip) {
            return;
        }
        SessionCheck sessionCheck = parser.parse(record);
        if (sessionCheck != null) {
            for (Long uid: sessionCheck.uid()) {
                SearchMapShard shard = context.shard(uid);

                AggregatedUser user =
                    writersMap.computeIfAbsent(
                        shard,
                        h -> new LinkedHashMap<>(DEFAULT_USERS_SIZE))
                        .computeIfAbsent(uid, u -> new AggregatedUser());
                long sts = sessionCheck.ts() * MILLS_IN_SEC;
                if (System.currentTimeMillis() - sts > MAX_DELAY) {
                    continue;
                }

                user.add(
                    sts,
                    sessionCheck.yuid());
            }
        }
    }

    private static final class AggregatedUser implements JsonValue {
        private volatile long ts;
        private volatile Set<String> yuids;

        private AggregatedUser() {
            this.ts = -1;
            this.yuids = new LinkedHashSet<>();
        }

        private AggregatedUser(final long ts, final Set<String> yuids) {
            this.ts = ts;
            this.yuids = new LinkedHashSet<>(yuids);
        }

        public void add(final long ts, final String yuid) {
            if (ts > this.ts) {
                this.ts = ts;
            }

            if (yuid != null) {
                this.yuids.add(yuid);
            }
        }

        public long ts() {
            return ts;
        }

        public Set<String> yuids() {
            return yuids;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("ts");
            writer.value(ts);
            writer.key("yuids");
            writer.value(yuids);
            writer.endObject();
        }
    }
}
