package ru.yandex.logbroker.log.consumer.blackbox;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.message.BasicHttpRequest;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logbroker.config.ImmutableLogbrokerConsumerServerConfig;
import ru.yandex.logbroker.config.LogConfig;
import ru.yandex.logbroker.config.blackbox.ImmutableOnlineUsersStorageConfig;
import ru.yandex.logbroker.config.blackbox.OnlineUsersStorageConfigBuilder;
import ru.yandex.logbroker.log.LogbrokerTskvRecord;
import ru.yandex.logbroker.log.consumer.LogConsumer;
import ru.yandex.logbroker.log.consumer.TskvLogConsumerFactory;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class BlackboxLogConsumerFactory
    extends AbstractStatable
    implements TskvLogConsumerFactory
{
    public static final String NAME = "blackbox";
    private static final String CLIENT_STAT_PATH = "/useronline/client";
    private static final String CLIENT_NAME = "useronline-client";

    private final AsyncClient userOnlineClient;
    private final ImmutableOnlineUsersStorageConfig userOnlineConfig;
    private final BlackboxConsumerContext context;

    public BlackboxLogConsumerFactory(
        final SharedConnectingIOReactor reactor,
        final ImmutableLogbrokerConsumerServerConfig serverConfig,
        final ImmutableOnlineUsersStorageConfig userOnlineConfig)
        throws ConfigException
    {
        this.userOnlineConfig = userOnlineConfig;
        userOnlineClient =
            new AsyncClient(
                reactor,
                userOnlineConfig,
                RequestErrorType.ERROR_CLASSIFIER)
                .adjustStater(
                    serverConfig.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            CLIENT_STAT_PATH)));

        TimeFrameQueue<Long> peQueue =
            new TimeFrameQueue<>(serverConfig.metricsTimeFrame());
        TimeFrameQueue<Long> staleQueue =
            new TimeFrameQueue<>(serverConfig.metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                peQueue,
                new NamedStatsAggregatorFactory<>(
                    NAME + "-consumer-parse-error_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        registerStater(
            new PassiveStaterAdapter<>(
                staleQueue,
                new NamedStatsAggregatorFactory<>(
                    NAME + "-consumer-stale-records_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.context =
            new BlackboxConsumerContext(
                staleQueue,
                peQueue,
                userOnlineClient,
                userOnlineConfig,
                serverConfig.loggers().preparedLoggers().get(
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            "/consumer")))
                    .addPrefix("BlackboxConsumer"));

        this.userOnlineClient.start();
    }

    public BlackboxLogConsumerFactory(
        final SharedConnectingIOReactor reactor,
        final ImmutableLogbrokerConsumerServerConfig serverConfig,
        final IniConfig config)
        throws ConfigException
    {
        this(
            reactor,
            serverConfig,
            new OnlineUsersStorageConfigBuilder(config).build());
    }

    @Override
    public LogConsumer<LogbrokerTskvRecord> create(
        final LogConfig config,
        final PrefixedLogger logger)
    {
        return new BlackboxLogConsumer(context, logger);
    }

    @Override
    public void close() throws IOException {
        this.userOnlineClient.close();
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put(CLIENT_NAME, userOnlineClient.status(verbose));
        return status;
    }

    public ImmutableOnlineUsersStorageConfig config() {
        return userOnlineConfig;
    }
}
