package ru.yandex.logbroker.log.consumer.integration;

import java.util.LinkedHashSet;
import java.util.Set;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.regex.Pattern;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logbroker.log.LogbrokerTskvRecord;
import ru.yandex.logbroker.log.consumer.integration.click.UserAction;
import ru.yandex.logbroker.log.consumer.integration.click.UserInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.tskv.TskvRecord;

public abstract class AbstractJsIntegrationLogParser
    implements Function<LogbrokerTskvRecord, UserAction>
{
    protected static final String THREAD = "thread";
    protected static final String TYPE = "type";
    private static final String FROM = "from";
    private static final String THREAD_PREFIX = "t";
    private static final Pattern MID_PATTERN = Pattern.compile("[t]?\\d+");

    protected final PrefixedLogger logger;
    protected final TimeFrameQueue<Long> parseErrors;
    protected final TimeFrameQueue<Long> potentialMissed;

    protected AbstractJsIntegrationLogParser(
        final PrefixedLogger logger,
        final TimeFrameQueue<Long> parseErrors,
        final TimeFrameQueue<Long> potentialMissed)
    {
        this.logger = logger;
        this.parseErrors = parseErrors;
        this.potentialMissed = potentialMissed;
    }

    protected UserInfo resolveUser(final TskvRecord record) {
        String uid = record.get("uid");

        String prefixStr = null;
        if (uid == null) {
            logger.log(
                Level.WARNING,
                "Skipping, no uid " + record.toString());
        } else {
            prefixStr = uid;
        }

        Prefix prefix = null;
        if (prefixStr != null) {
            try {
                prefix = PrefixType.LONG.parse(prefixStr);
            } catch (java.text.ParseException pe) {
                logger.log(
                    Level.WARNING,
                    "Unable to parse prefix "
                        + prefixStr + ' ' + record.toString(),
                    pe);
            }
        }

        Long uidL = null;
        try {
            if (uid != null) {
                uidL = Long.parseLong(uid);
            }
        } catch (NumberFormatException nfe) {
            logger.log(
                Level.WARNING,
                "Unable to parse user id " + record.toString(),
                nfe);
        }

        if (prefix == null) {
            logger.info(
                "Can not resolve user, no prefix or uid or suid "
                    + record.toString());
            return null;
        }

        return new UserInfo(uidL, new User("pg", prefix));
    }

    // CSOFF: ReturnCount
    protected UserAction parseFilterAndMessage(
        final UserAction click,
        final JsonMap message,
        final JsonMap filter)
        throws JsonException
    {
        String type = filter.getString(TYPE, null);
        String param = filter.getString("param", null);

        if (type != null && param != null) {
            if (THREAD.equals(type)) {
                return null;
            }

            if ("search".equalsIgnoreCase(type)) {
                CgiParams params = new CgiParams(param);
                String request = params.getString("request", null);
                click.search(true);
                click.request(request);
            } else {
                click.search(false);
            }
        } else {
            click.search(false);
        }

        if (message.containsKey(TYPE)) {
            Set<Integer> mtypes = new LinkedHashSet<>();
            for (JsonObject mtypeObj: message.getList(TYPE)) {
                mtypes.add((int) mtypeObj.asLong());
            }

            click.mtypes(mtypes);
        }

        click.from(message.getString(FROM, null));

        String mid = message.getString("mid");
        if (!MID_PATTERN.matcher(mid).matches()) {
            logger.warning("Invalid mid found " + message.toString());
            parseErrors.accept(1L);
            return null;
        }

        if (mid.startsWith(THREAD_PREFIX)) {
            click.mid(mid.replace(THREAD_PREFIX, ""));
        } else {
            click.mid(mid);
        }

        click.thread(message.getLong(THREAD, -1L));
        click.index(message.getLong("index", -1L));

        return click;
    }
    // CSON: ReturnCount
}
