package ru.yandex.logbroker.log.consumer.integration;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.logbroker.config.LogConfig;
import ru.yandex.logbroker.config.integration.ClickIndexerConfigBuilder;
import ru.yandex.logbroker.config.integration.ImmutableClickIndexerConfig;
import ru.yandex.logbroker.log.LogbrokerTskvRecord;
import ru.yandex.logbroker.log.consumer.LogConsumer;
import ru.yandex.logbroker.log.consumer.TskvLogConsumerFactory;
import ru.yandex.logbroker.log.consumer.integration.classification.ClassificationLogConsumerFactory;
import ru.yandex.logbroker.log.consumer.integration.click.IndexationClient;
import ru.yandex.logbroker.log.consumer.integration.click.UserAction;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.StatsConsumer;

public class IntegrationLogConsumerFactory
    extends AbstractStatable
    implements TskvLogConsumerFactory
{
    public static final String NAME = "integration";

    private static final String CLIENT_NAME = "index-client";

    private final IndexationClient indexClient;
    private final NotifyLogConsumerFactory notifyFactory;
    private final UpdateLogConsumerFactory updateFactory;
    private final ClassificationLogConsumerFactory classificationFactory;
    private final Function<LogbrokerTskvRecord, UserAction> tskvParser;
    private final TimeFrameQueue<Long> parseErrors;
    private final TimeFrameQueue<Long> potentialMissing;
    private final TimeFrameQueue<Long> clicks;

    // CSOFF: ParameterNumber
    public IntegrationLogConsumerFactory(
        final SharedConnectingIOReactor reactor,
        final ImmutableBaseServerConfig serverConfig,
        final ImmutableHttpHostConfig indexConfig,
        final ImmutableClickIndexerConfig notifyConfig,
        final ImmutableClickIndexerConfig updateConfig,
        final ImmutableClickIndexerConfig classificationConfig)
        throws ConfigException
    {
        indexClient =
            new IndexationClient(reactor, indexConfig)
                .adjustStater(
                    serverConfig.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            "/index/client")));
        PrefixedLogger logger =
            serverConfig.loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/consumer")));
        IntegrationContext context =
            new IntegrationContext(indexClient, indexConfig);
        context.serverConfig(serverConfig);
        context.logger(logger);

        this.notifyFactory =
            new NotifyLogConsumerFactory(context, notifyConfig);
        this.updateFactory =
            new UpdateLogConsumerFactory(context, updateConfig);
        this.classificationFactory =
            new ClassificationLogConsumerFactory(context, classificationConfig);

        parseErrors =
            new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());
        potentialMissing =
            new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());
        clicks =
            new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                parseErrors,
                new NamedStatsAggregatorFactory<>(
                    "integration-consumer-parse-error_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        registerStater(
            new PassiveStaterAdapter<>(
                potentialMissing,
                new NamedStatsAggregatorFactory<>(
                    "integration-consumer-potential-missing_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        registerStater(
            new PassiveStaterAdapter<>(
                clicks,
                new NamedStatsAggregatorFactory<>(
                    "integration-consumer-clicks_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.tskvParser =
            new TskvJsIntegrationLogParser(
                logger,
                parseErrors,
                potentialMissing);

        this.indexClient.start();
    }
    // CSON: ParameterNumber

    public IntegrationLogConsumerFactory(
        final SharedConnectingIOReactor reactor,
        final ImmutableBaseServerConfig serverConfig,
        final IniConfig config)
        throws ConfigException
    {
        this(
            reactor,
            serverConfig,
            new HttpHostConfigBuilder(config).build(),
            new ClickIndexerConfigBuilder(config.section("notify")).build(),
            new ClickIndexerConfigBuilder(config.section("update")).build(),
            new ClickIndexerConfigBuilder(config.section("classification"))
                .build());
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        super.stats(statsConsumer);

        notifyFactory.stats(statsConsumer);
        updateFactory.stats(statsConsumer);
        classificationFactory.stats(statsConsumer);
    }

    @Override
    public LogConsumer<LogbrokerTskvRecord> create(
        final LogConfig config,
        final PrefixedLogger logger)
    {
        return new MultiLogConsumer(
            notifyFactory.create(config),
            updateFactory.create(config),
            classificationFactory.create(config));
    }

    private final class MultiLogConsumer
        implements LogConsumer<LogbrokerTskvRecord>
    {
        private final BiFunction<UserAction, FutureCallback<Object>, Object>
            notify;
        private final BiFunction<UserAction, FutureCallback<Object>, Object>
            update;
        private final BiFunction<UserAction, FutureCallback<Object>, Object>
            classification;

        private MultiLogConsumer(
            final BiFunction<UserAction, FutureCallback<Object>, Object> notify,
            final BiFunction<UserAction, FutureCallback<Object>, Object> update,
            final BiFunction<UserAction, FutureCallback<Object>, Object>
                classification)
        {
            this.notify = notify;
            this.update = update;
            this.classification = classification;
        }

        @Override
        public void apply(
            final LogbrokerTskvRecord record,
            final FutureCallback<Object> callback)
        {
            UserAction click = tskvParser.apply(record);

            if (click == null) {
                callback.completed(null);
                return;
            }

            clicks.accept(1L);
            if (click.actionType() == UserAction.ActionType.SHOW) {
                DoubleFutureCallback<Object, Object> cb =
                    new DoubleFutureCallback<>(callback);

                notify.apply(click, cb.first());
                update.apply(click, cb.second());
            } else if (click.actionType()
                == UserAction.ActionType.CLASSIFICATION)
            {
                classification.apply(click, callback);
            } else {
                callback.completed(null);
            }
        }
    }

    @Override
    public void close() throws IOException {
        indexClient.close();
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put(CLIENT_NAME, indexClient.status(verbose));
        return status;
    }
}
