package ru.yandex.logbroker.log.consumer.integration;

import java.io.IOException;
import java.util.Locale;
import java.util.function.BiFunction;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.PgFields;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.RetryContextFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.logbroker.config.LogConfig;
import ru.yandex.logbroker.config.integration.ClickIndexerConfigBuilder;
import ru.yandex.logbroker.config.integration.ImmutableClickIndexerConfig;
import ru.yandex.logbroker.log.consumer.integration.click.IndexationClient;
import ru.yandex.logbroker.log.consumer.integration.click.LinearIntervalRetryContextFactory;
import ru.yandex.logbroker.log.consumer.integration.click.LoggableProducerSupplier;
import ru.yandex.logbroker.log.consumer.integration.click.UserAction;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class NotifyLogConsumerFactory extends AbstractStatable {
    protected static final String CHANGE_TYPE =
        ChangeType.FIELDS_UPDATE.name().toLowerCase(Locale.ENGLISH);

    protected static final String LOG_PREFIX = "NotifyConsumer";
    protected static final String PREFIX_PARAM = "prefix";

    private final ImmutableClickIndexerConfig config;
    private final IndexationClient indexClient;
    private final PrefixedLogger logger;
    private final HttpHost indexationHost;

    private final TimeFrameQueue<Long> indexed;
    private final TimeFrameQueue<Long> nonPg;
    private final TimeFrameQueue<Long> error;

    public NotifyLogConsumerFactory(
        final IntegrationContext context,
        final IniConfig config)
        throws ConfigException
    {
        this(context, new ClickIndexerConfigBuilder(config).build());
    }

    public NotifyLogConsumerFactory(
        final IntegrationContext context,
        final ImmutableClickIndexerConfig config)
    {
        this.config = config;
        this.logger = context.logger().addPrefix(LOG_PREFIX);
        this.indexClient = context.indexClient();
        this.indexationHost = context.indexClientConfig().host();

        indexed =
            new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());
        nonPg = new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());
        error = new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                nonPg,
                new NamedStatsAggregatorFactory<>(
                    "integration-notify-consumer-nonpg_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        registerStater(
            new PassiveStaterAdapter<>(
                error,
                new NamedStatsAggregatorFactory<>(
                    "integration-notify-consumer-failed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        registerStater(
            new PassiveStaterAdapter<>(
                indexed,
                new NamedStatsAggregatorFactory<>(
                    "integration-notify-consumer-indexed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public BiFunction<UserAction, FutureCallback<Object>, Object> create(
        final LogConfig config)
    {
        return new NotifyLogConsumer();
    }

    private final class NotifyLogConsumer
        implements BiFunction<UserAction, FutureCallback<Object>, Object>
    {
        String uri(final UserAction click) {
            QueryConstructor uri = new QueryConstructor(config.indexUri());

            try {
                String service;
                if (click.userInfo().corp()) {
                    service = config.pgCorpQueue();
                } else {
                    service = config.pgQueue();
                }

                uri.append("service", service);

                uri.append(
                    PREFIX_PARAM,
                    click.userInfo().user().prefix().toString());
                uri.append("mdb", click.userInfo().user().service());
                uri.append("logbroker", "click_consumer");
            } catch (BadRequestException bre) {
                logger.log(
                    Level.SEVERE,
                    "Unable construct producer request " + click.toString(),
                    bre);
                return null;
            }

            return uri.toString();
        }

        String body(final UserAction click) {
            StringBuilderWriter sbWriter = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(sbWriter)) {
                writer.startObject();
                writer.key(PgFields.UID);
                writer.value(click.userInfo().user().prefix());
                writer.key(PgFields.CHANGE_TYPE);
                writer.value(CHANGE_TYPE);
                writer.key("operation_date");
                writer.value((long) click.timestamp());
                writer.key(PgFields.CHANGED);
                writer.startArray();
                writer.startObject();
                writer.key(PgFields.MID);
                writer.value(click.mid());

                if (click.search()) {
                    incCounter("clicks_serp_count", writer);
                }

                incCounter("clicks_total_count", writer);

                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                logger.log(
                    Level.SEVERE,
                    "Can not create json " + click.toString(),
                    ioe);
                return null;
            }

            return sbWriter.toString();
        }

        private void incCounter(final String name, final JsonWriterBase writer)
            throws IOException
        {
            writer.key(name);

            writer.startObject();
            writer.key("function");
            writer.value("inc");
            writer.endObject();
        }

        @Override
        public Object apply(
            final UserAction click,
            final FutureCallback<Object> callback)
        {
            boolean enabled = true;
            if (!"pg".equalsIgnoreCase(click.userInfo().user().service())) {
                nonPg.accept(1L);
                enabled = false;
            }

            String uri = uri(click);
            if (uri == null) {
                error.accept(1L);
                enabled = false;
            }

            if (!enabled) {
                callback.completed(null);
                return null;
            }

            String content = body(click);

            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(uri, content);
            RetryContextFactory retryContextFactory =
                new LinearIntervalRetryContextFactory(
                    new LoggableProducerSupplier<>(
                        producerGenerator,
                        indexationHost,
                        logger));

            indexClient.execute(
                retryContextFactory,
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                indexClient.httpClientContextGenerator(),
                new NotifyIndexCallback(callback));
            return null;
        }

        private final class NotifyIndexCallback
            implements FutureCallback<HttpResponse>
        {
            private final FutureCallback<Object> callback;

            private NotifyIndexCallback(
                final FutureCallback<Object> callback)
            {
                this.callback = callback;
            }

            @Override
            public void completed(final HttpResponse response) {
                callback.completed(response);
                indexed.accept(1L);
            }

            @Override
            public void failed(final Exception e) {
                logger.log(Level.WARNING, "Notify failed", e);
                callback.failed(e);
                error.accept(1L);
            }

            @Override
            public void cancelled() {
                callback.cancelled();
            }
        }
    }
}
