package ru.yandex.logbroker.log.consumer.integration;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logbroker.log.LogbrokerTskvRecord;
import ru.yandex.logbroker.log.consumer.integration.click.UserAction;
import ru.yandex.logbroker.log.consumer.integration.click.UserInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.IntegerParser;
import ru.yandex.tskv.TskvException;

public class TskvJsIntegrationLogParser extends AbstractJsIntegrationLogParser {
    public TskvJsIntegrationLogParser(
        final PrefixedLogger logger,
        final TimeFrameQueue<Long> parseErrors,
        final TimeFrameQueue<Long> parseMissing)
    {
        super(logger, parseErrors, parseMissing);
    }

    // CSOFF: ReturnCount
    @Override
    public UserAction apply(final LogbrokerTskvRecord record) {
        String type = record.get(TYPE);
        if (type != null && type.equalsIgnoreCase("YANDEXUID_WRONG")) {
            return null;
        }

        UserInfo user = resolveUser(record);
        if (user == null) {
            return null;
        }

        String actionType = record.get("action");
        if ("classification".equalsIgnoreCase(actionType)) {
            logger.info("Classification " + record.toString());
            UserAction classAction =
                new UserAction(
                    user,
                    UserAction.ActionType.CLASSIFICATION);
            String mid = record.getString("mid", null);
            if (mid == null) {
                logger.warning(
                    "Missing mid for Classification " + record.toString());
                parseErrors.accept(1L);
                return null;
            }

            String target = record.getString("target", "");
            Set<Integer> types;
            try {
                types = record.get(
                    "types",
                    Collections.emptySet(),
                    new CollectionParser<>(
                        IntegerParser.INSTANCE,
                        LinkedHashSet::new));
            } catch (TskvException te) {
                logger.log(
                    Level.WARNING,
                    "Bad types in record " + record.toString(),
                    te);
                types = Collections.emptySet();
            }

            classAction.target(target);
            classAction.mid(mid);
            classAction.timestamp(record.unixtime());
            classAction.mtypes(types);
            return classAction;
        }

        if (actionType == null || !actionType.equalsIgnoreCase("show")) {
            return null;
        }

        UserAction click = new UserAction(user, UserAction.ActionType.SHOW);

        click.timestamp(record.unixtime());

        String filterStr = record.get("filter");
        String messageStr = record.get("message");

        if (filterStr == null) {
            logger.warning(
                "No filter fields in " + record.toString());
            potentialMissed.accept(1L);
            return null;
        }

        try {
            JsonMap filter =
                TypesafeValueContentHandler.parse(filterStr).asMap();
            if (THREAD.equalsIgnoreCase(
                filter.getString(TYPE, null)))
            {
                //https://st.yandex-team.ru/PS-2946
                String mid = filter.getString("param");
                if (mid == null || mid.length() < 2) {
                    parseErrors.accept(1L);
                    return null;
                }

                long midL = Long.parseLong(mid.substring(1));
                click.search(false);
                click.mid(String.valueOf(midL));
                click.thread(midL);
            } else {
                if (messageStr == null || record.get("messages") != null) {
                    logger.warning(
                        "No message or messages instead of message, skipping");
                    potentialMissed.accept(1L);
                    return null;
                }

                JsonMap message =
                    TypesafeValueContentHandler.parse(messageStr).asMap();
                click = parseFilterAndMessage(click, message, filter);
            }
        } catch (JsonException | NumberFormatException e) {
            logger.log(
                Level.WARNING,
                "Failed parse click " + record.toString(),
                e);
            parseErrors.accept(1L);
            return null;
        }

        return click;
    }
    // CSON: ReturnCount
}
