package ru.yandex.logbroker.log.consumer.integration;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.function.BiFunction;
import java.util.logging.Level;

import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.RetryContextFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logbroker.config.LogConfig;
import ru.yandex.logbroker.config.integration.ClickIndexerConfigBuilder;
import ru.yandex.logbroker.config.integration.ImmutableClickIndexerConfig;
import ru.yandex.logbroker.log.consumer.integration.click.IndexationClient;
import ru.yandex.logbroker.log.consumer.integration.click.LinearIntervalRetryContextFactory;
import ru.yandex.logbroker.log.consumer.integration.click.LoggableProducerSupplier;
import ru.yandex.logbroker.log.consumer.integration.click.UserAction;
import ru.yandex.logbroker.log.consumer.integration.mtype.MtypeClickConsumer;
import ru.yandex.logbroker.log.consumer.integration.requests.RequestsClickConsumer;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.StatsConsumer;

public class UpdateLogConsumerFactory
    extends AbstractStatable
{
    private static final String LOG_PREFIX = "UpdateConsumer";
    private static final String PREFIX_PARAM = "prefix";

    protected final ImmutableClickIndexerConfig config;
    protected final ImmutableHttpHostConfig indexClientConfig;

    private final List<SlaveClickConsumer> slaves;
    private final IndexationClient producerClient;

    private final PrefixedLogger logger;

    private final TimeFrameQueue<Long> skip;
    private final TimeFrameQueue<Long> ok;
    private final TimeFrameQueue<Long> fail;

    public UpdateLogConsumerFactory(
        final IntegrationContext context,
        final IniConfig config)
        throws ConfigException
    {
        this(context, new ClickIndexerConfigBuilder(config).build());
    }

    public UpdateLogConsumerFactory(
        final IntegrationContext context,
        final ImmutableClickIndexerConfig config)
        throws ConfigException
    {
        this.config = config;
        this.producerClient = context.indexClient();
        this.indexClientConfig = context.indexClientConfig();
        this.logger = context.logger().addPrefix(LOG_PREFIX);
        this.slaves =
            Arrays.asList(
                new MtypeClickConsumer(context.serverConfig()),
                new RequestsClickConsumer(context.serverConfig()));

        ok = new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());
        skip = new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());
        fail = new TimeFrameQueue<>(context.serverConfig().metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                ok,
                new NamedStatsAggregatorFactory<>(
                    "integration-update-total-consumer-ok_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        registerStater(
            new PassiveStaterAdapter<>(
                skip,
                new NamedStatsAggregatorFactory<>(
                    "integration-update-total-consumer-skip_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        registerStater(
            new PassiveStaterAdapter<>(
                fail,
                new NamedStatsAggregatorFactory<>(
                    "integration-update-total-consumer-fail_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public BiFunction<UserAction, FutureCallback<Object>, Object> create(
        final LogConfig config)
    {
        return new UpdateLogConsumer();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        super.stats(statsConsumer);

        for (SlaveClickConsumer slave: slaves) {
            slave.stats(statsConsumer);
        }
    }

    private final class UpdateLogConsumer
        implements BiFunction<UserAction, FutureCallback<Object>, Object>
    {
        @Override
        public Object apply(
            final UserAction click,
            final FutureCallback<Object> callback)
        {
            boolean enabled = true;
            if (!"pg".equalsIgnoreCase(click.userInfo().user().service())) {
                enabled = false;
            }

            if (slaves.stream().noneMatch(v -> v.enabled(click))) {
                enabled = false;
            }

            String uri = uri(click);
            if (uri == null) {
                enabled = false;
            }

            IndexCallback indexCallback = body(click, callback);
            if (indexCallback == null) {
                enabled = false;
            }

            if (!enabled) {
                slaves.forEach(SlaveClickConsumer::skip);
                skip.accept(1L);
                callback.completed(null);
                return null;
            }

            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    uri,
                    indexCallback.content());

            RetryContextFactory retryContextFactory =
                new LinearIntervalRetryContextFactory(
                    new LoggableProducerSupplier<>(
                        producerGenerator,
                        indexClientConfig.host(),
                        logger));

            producerClient.execute(
                retryContextFactory,
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                producerClient.httpClientContextGenerator(),
                indexCallback);

            return null;
        }

        private IndexCallback body(
            final UserAction click,
            final FutureCallback<Object> callback)
        {
            StringBuilderWriter sbWriter = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(sbWriter)) {
                writer.startObject();
                writer.key(PREFIX_PARAM);
                writer.value(click.prefix());
                writer.key("AddIfNotExists");
                writer.value(true);
                writer.key("docs");
                writer.startArray();

                for (SlaveClickConsumer slave : slaves) {
                    if (slave.enabled(click)) {
                        slave.write(writer, click);
                    }
                }

                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                logger.log(
                    Level.WARNING,
                    "Failed to construct body for click",
                    ioe);
                return null;
            }

            return new IndexCallback(callback, sbWriter.toString(), slaves);
        }

        private String uri(final UserAction click) {
            QueryConstructor uri = new QueryConstructor(config.indexUri());

            try {
                String service;
                if (click.userInfo().corp()) {
                    service = config.pgCorpQueue();
                } else {
                    service = config.pgQueue();
                }

                uri.append("service", service);
                uri.append(
                    PREFIX_PARAM,
                    click.prefix().toString());
                uri.append("logbroker", "update_consumer");
            } catch (BadRequestException bre) {
                logger.log(
                    Level.SEVERE,
                    "Unable construct producer request " + click.toString(),
                    bre);
                return null;
            }

            return uri.toString();
        }

        private final class IndexCallback
            implements FutureCallback<HttpResponse>
        {
            private final FutureCallback<Object> callback;
            private final String content;
            private final List<SlaveClickConsumer> slaves;

            private IndexCallback(
                final FutureCallback<Object> callback,
                final String content,
                final List<SlaveClickConsumer> salves)
            {
                this.callback = callback;
                this.content = content;
                this.slaves = salves;
            }

            public String content() {
                return content;
            }

            @Override
            public void completed(final HttpResponse response) {
                callback.completed(response);
                ok.accept(1L);

                slaves.forEach(SlaveClickConsumer::indexed);
            }

            @Override
            public void failed(final Exception e) {
                callback.failed(e);
                logger.log(Level.WARNING, "Update failed " + content, e);
                fail.accept(1L);
                slaves.forEach(SlaveClickConsumer::failed);
            }

            @Override
            public void cancelled() {
                callback.cancelled();
            }
        }
    }
}
