package ru.yandex.logbroker.log.consumer.integration.classification;

import java.io.IOException;
import java.util.Locale;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiFunction;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.dbfields.PgFields;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.RetryContextFactory;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logbroker.config.LogConfig;
import ru.yandex.logbroker.config.integration.ImmutableClickIndexerConfig;
import ru.yandex.logbroker.log.consumer.integration.IntegrationContext;
import ru.yandex.logbroker.log.consumer.integration.click.IndexationClient;
import ru.yandex.logbroker.log.consumer.integration.click.LinearIntervalRetryContextFactory;
import ru.yandex.logbroker.log.consumer.integration.click.LoggableProducerSupplier;
import ru.yandex.logbroker.log.consumer.integration.click.UserAction;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class ClassificationLogConsumerFactory extends AbstractStatable {
    protected static final String CHANGE_TYPE =
        ChangeType.FIELDS_UPDATE.name().toLowerCase(Locale.ENGLISH);
    protected static final String LOG_PREFIX = "ClassificationConsumer";
    protected static final String PREFIX_PARAM = "prefix";

    private final TimeFrameQueue<Long> indexed;
    private final TimeFrameQueue<Long> error;

    private final IndexationClient indexClient;
    private final HttpHost indexationHost;
    private final PrefixedLogger logger;

    private final ImmutableBaseServerConfig serverConfig;
    private final ConcurrentHashMap<Integer, TimeFrameQueue<Long>> typesSigMap;

    public ClassificationLogConsumerFactory(
        final IntegrationContext context,
        final ImmutableClickIndexerConfig config)
    {
        serverConfig = context.serverConfig();
        indexed =
            new TimeFrameQueue<>(serverConfig.metricsTimeFrame());
        error = new TimeFrameQueue<>(serverConfig.metricsTimeFrame());

        this.indexClient = context.indexClient();
        this.indexationHost = context.indexClientConfig().host();
        this.logger = context.logger().addPrefix(LOG_PREFIX);

        this.typesSigMap = new ConcurrentHashMap<>();

        registerStater(
            new PassiveStaterAdapter<>(
                indexed,
                new NamedStatsAggregatorFactory<>(
                    "integration-classification-consumer-clicks_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        registerStater(
            new PassiveStaterAdapter<>(
                error,
                new NamedStatsAggregatorFactory<>(
                    "integration-classification-consumer-failed_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public BiFunction<UserAction, FutureCallback<Object>, Object> create(
        final LogConfig config)
    {
        return new ClassificationLogConsumer();
    }

    private final class ClassificationLogConsumer
        implements BiFunction<UserAction, FutureCallback<Object>, Object>
    {
        String uri(final UserAction click) {
            QueryConstructor uri = new QueryConstructor("/notify?");

            try {
                uri.append("service", "change_log");

                uri.append(
                    PREFIX_PARAM,
                    click.userInfo().user().prefix().toString());
                uri.append("mdb", click.userInfo().user().service());
                uri.append("logbroker", "classification");
            } catch (BadRequestException bre) {
                logger.log(
                    Level.SEVERE,
                    "Unable construct producer request " + click.toString(),
                    bre);
                return null;
            }

            return uri.toString();
        }

        String body(final UserAction click) {
            StringBuilderWriter sbWriter = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(sbWriter)) {
                writer.startObject();
                writer.key(PgFields.UID);
                writer.value(click.userInfo().user().prefix());
                writer.key(PgFields.CHANGE_TYPE);
                writer.value(CHANGE_TYPE);
                writer.key("operation_date");
                writer.value((long) click.timestamp());
                writer.key(PgFields.CHANGED);
                writer.startArray();
                writer.startObject();
                writer.key(PgFields.MID);
                writer.value(click.mid());
                writer.key(MailIndexFields.CLASSIFICATION_CLICK);
                writer.value(true);
                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                logger.log(
                    Level.SEVERE,
                    "Can not create json " + click.toString(),
                    ioe);
                return null;
            }

            return sbWriter.toString();
        }

        @Override
        public Object apply(
            final UserAction click,
            final FutureCallback<Object> callback)
        {
            long signalValue = 0;
            if ("yes".equalsIgnoreCase(click.target())) {
                signalValue = 1;
            }

            for (Integer mtype: click.mtypes()) {
                TimeFrameQueue<Long> tfq =
                    typesSigMap.computeIfAbsent(mtype, (key) -> {
                        String postfix = mtype + "_ammm";
                        String clicks =
                            "integration-classification-consumer-clicks-"
                                + postfix;

                        String yesClicks =
                            "integration-classification-consumer-clicks-yes-"
                                + postfix;
                        TimeFrameQueue<Long> result =
                            new TimeFrameQueue<>(serverConfig
                                .metricsTimeFrame());
                        registerStater(
                            new PassiveStaterAdapter<>(
                                result,
                                new DuplexStaterFactory<>(
                                    new NamedStatsAggregatorFactory<>(
                                        yesClicks,
                                        IntegralSumAggregatorFactory.INSTANCE),
                                    new NamedStatsAggregatorFactory<>(
                                        clicks,
                                        CountAggregatorFactory.INSTANCE))));
                        return result;
                    });

                tfq.accept(signalValue);
            }

            boolean enabled = true;
            String uri = uri(click);
            if (uri == null) {
                error.accept(1L);
                enabled = false;
            }

            if (!enabled) {
                callback.completed(null);
                return null;
            }

            String content = body(click);

            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(uri, content);
            RetryContextFactory retryContextFactory =
                new LinearIntervalRetryContextFactory(
                    new LoggableProducerSupplier<>(
                        producerGenerator,
                        indexationHost,
                        logger));

            indexClient.execute(
                retryContextFactory,
                BasicAsyncResponseConsumerFactory.ANY_GOOD,
                indexClient.httpClientContextGenerator(),
                new ClassificationIndexCallback(callback));
            return null;
        }
    }

    private final class ClassificationIndexCallback
        implements FutureCallback<HttpResponse>
    {
        private final FutureCallback<Object> callback;

        private ClassificationIndexCallback(
            final FutureCallback<Object> callback)
        {
            this.callback = callback;
        }

        @Override
        public void completed(final HttpResponse response) {
            indexed.accept(1L);
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            error.accept(1L);
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            error.accept(1L);
            callback.cancelled();
        }
    }
}
