package ru.yandex.logbroker.server;

import java.util.List;
import java.util.logging.Level;

import ru.yandex.logbroker.client.OffsetInfo;
import ru.yandex.logbroker.client.Partition;
import ru.yandex.logbroker.client.ReadSession;
import ru.yandex.logbroker.client.Session;
import ru.yandex.logbroker.config.ImmutableTopicConfig;
import ru.yandex.logger.PrefixedLogger;

public class LogSkipUtil {
    private static final int RETRIES_COUNT = 30;
    private static final long RETRY_PERIOD = 1000;
    private static final int OFFSET_GAP = 10000;

    private final ImmutableTopicConfig context;

    public LogSkipUtil(final ImmutableTopicConfig config) {
        this.context = config;
    }

    public void skipLog(
        final List<Partition> partitions,
        final List<OffsetInfo> offsets)
        throws InterruptedException
    {
        PrefixedLogger logger = context.logger();
        for (OffsetInfo offsetInfo: offsets) {
            if (
                offsetInfo.logSize() <= 2 * OFFSET_GAP
                    || offsetInfo.lag() <= 2 * OFFSET_GAP)
            {
                logger.info(offsetInfo.toString() + " to small log size, done");
                continue;
            }

            Partition partition = null;
            for (Partition p: partitions) {
                if (p.idString().equalsIgnoreCase(
                    offsetInfo.partition().idString()))
                {
                    partition = p;
                    break;
                }
            }

            if (partition == null) {
                logger.warning(
                    "Missing partition " + offsetInfo.partition().idString());
                continue;
            }

            long commitOffset = offsetInfo.logSize() - OFFSET_GAP;
            logger.info("Skipping to end for " + partition);
            logger.info("Current offset is " + offsetInfo.offset());
            logger.info("Moving to " + commitOffset);
            int attempts = 0;
            while (attempts < RETRIES_COUNT) {
                try {
                    Session session = context.client().session(partition);
                    ReadSession readSession = session.read();
                    readSession.close();

                    session.commit(commitOffset);
                } catch (Exception e) {
                    logger.log(
                        Level.WARNING,
                        "Failed to skip partition " + partition.idString(),
                        e);
                    Thread.sleep(RETRY_PERIOD);
                } finally {
                    attempts += 1;
                }
            }
        }
    }
}
