package ru.yandex.logbroker.topic;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.logging.Level;

import ru.yandex.logbroker.Stoppable;
import ru.yandex.logbroker.client.Partition;
import ru.yandex.logbroker.client.ReadSession;
import ru.yandex.logbroker.client.Session;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.client.exception.LogbrokerBadRequestException;
import ru.yandex.logbroker.client.exception.LogbrokerConflictException;
import ru.yandex.logbroker.config.ImmutableTopicConfig;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.tskv.TskvRecord;

public class PartitionWorker
    extends AbstractStatable
    implements Runnable, Closeable, Stoppable
{
    private static final int NEXT_LINE = 13;
    private static final int REV = 10;
    private static final int THOUSAND = 1000;

    protected final ImmutableTopicConfig config;
    protected final Partition partition;
    protected final PrefixedLogger logger;
    protected final ChunkTskvParser parser;

    private volatile boolean stop = false;
    private volatile boolean stopped = true;
    private long lastCommit;
    private long lastReadOffset;
    private long chunksProcessed;

    private final ReadMetrics readMetrics;

    public PartitionWorker(
        final ImmutableTopicConfig topicConfig,
        final Partition partition,
        final ReadMetrics readMetrics)
    {
        this.config = topicConfig;
        this.partition = partition;
        this.readMetrics = readMetrics;

        if (prefix().isEmpty()) {
            this.logger = config.logger();
        } else {
            this.logger = config.logger().addPrefix(prefix());
        }

        this.parser = new ChunkTskvParser(Charset.forName("utf-8"));
    }

    protected String prefix() {
        return "";
    }

    @Override
    public void close() throws IOException {
        this.stop = true;
    }

    @Override
    public boolean stopped() {
        return stopped;
    }

    @Override
    public void stop() {
        try {
            this.close();
        } catch (IOException ioe) {
            logger.log(Level.WARNING, "Close exception", ioe);
        }
    }

    private Session createSession(final Partition p) {
        Session result = null;
        try {
            result = config.client().session(p);
        } catch (LogbrokerConflictException lfe) {
            logger.log(Level.WARNING, "Conflict for " + p, lfe);
        } catch (LogbrokerBadRequestException lbe) {
            logger.log(Level.WARNING, "Bad request " + p, lbe);
        } catch (LogbrockerClientException lce) {
            logger.log(Level.WARNING, "Retrieve session error", lce);
        }

        return result;
    }

    protected DataChunk fetchChunk(
        final ReadSession session)
        throws IOException
    {
        long secondsAfterCommit =
            (System.currentTimeMillis() - lastCommit) / THOUSAND;

        byte[] chunk = session.stream().readChunk();
        if (chunk == null) {
            logger.info(
                "Limit reached, seconds: "
                    + secondsAfterCommit
                    + " chunks " + chunksProcessed);
            return null;
        }
        //logger.info("Chunk read " + chunk.length);
        int pos = -1;
        for (int i = 0; i < chunk.length; i++) {
            if (chunk[i] == NEXT_LINE) {
                if (i + 1 < chunk.length && chunk[i + 1] == REV) {
                    continue;
                }
            }

            if (chunk[i] == NEXT_LINE || chunk[i] == REV) {
                pos = i;
                break;
            }
        }

        if (pos <= 0) {
            throw new IOException("Broken chunk");
        }

        TskvRecord record = parser.parse(chunk, 0, pos + 1);

        ByteArrayInputStream data =
            new ByteArrayInputStream(chunk, pos + 1, chunk.length - pos - 1);
        DataChunk dataChunk =
            new DataChunk(
                record,
                session.session().partition().idString(),
                session.session().charset(),
                data);

        return dataChunk;
    }

    protected void consumeChunk(final DataChunk chunk) {
        config.consumer().chunk(chunk);
        lastReadOffset = chunk.offset();
        chunksProcessed += 1;
        readMetrics.chunkRead();
    }

    private void processRead(
        final ReadSession session)
        throws IOException
    {
        boolean first = true;

        while (!stop) {
            DataChunk chunk = fetchChunk(session);
            if (chunk == null) {
                return;
            }

            if (first) {
                logger.info("Starting with " + chunk.header().toString());
                first = false;
            }

            if (stop) {
                break;
            }

            consumeChunk(chunk);
        }
    }

    private void commit(final Session session) {
        long consumedOffset =
            config.consumer().lastOffset(partition.id());
        if (chunksProcessed > 0 && lastReadOffset >= 0) {
            try {
                logger.fine(
                    "Commit " + session
                        + " chunks read " + chunksProcessed
                        + " last read offset " + lastReadOffset
                        + " consumed offset " + consumedOffset);

                session.commit(lastReadOffset);
                chunksProcessed = 0;
                lastCommit = System.currentTimeMillis();
            } catch (LogbrockerClientException lbe) {
                logger.log(
                    Level.WARNING,
                    "Unable to commit " + lastReadOffset,
                    lbe);
            }
        }
    }

    private void consume(final Session session) {
        lastReadOffset = -1;

        ReadSession readSession = null;
        while (!stop) {
            try {
                readSession = session.read();

                logger.fine("Got session " + session.id());
                processRead(readSession);
            } catch (LogbrockerClientException lbe) {
                logger.log(Level.WARNING, "Read error", lbe);
                break;
            } catch (IOException ioe) {
                logger.log(
                    Level.WARNING,
                    "Read error io error " + session, ioe);
                break;
            } finally {
                try {
                    logger.info("Closing session" + session.id());
                    if (readSession != null) {
                        readSession.close();
                        readSession = null;
                    }
                } catch (IOException ioe) {
                    logger.log(
                        Level.WARNING,
                        "Unable close reader",
                        ioe);
                } finally {
                    commit(session);
                }
            }
        }

        logger.info("Consume done" + session.id());
    }

    @Override
    public void run() {
        this.stopped = false;
        Session session = null;
        //BufferedContentHandler contentHandler = new BufferedContentHandler();
        logger.info("Consuming partition: " + partition);
        try {
            session = createSession(partition);
            if (session == null) {
                logger.log(
                    Level.WARNING,
                    "Unable make session, retry later " + partition);
                Thread.sleep(
                    config.config().clientConfig().sessionRetryDelay());
                return;
            }

            logger.fine("Session established " + session);
            lastCommit = System.currentTimeMillis();

            consume(session);

            //contentHandler.read(session);
            logger.fine("Read completed " + session);
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Interrupted", ie);
        } catch (Exception e) {
            logger.log(Level.SEVERE, "Exception in worker", e);
            throw e;
        } finally {
            logger.log(Level.INFO, "Worker finished " + partition);

            try {
                if (session != null) {
                    session.close();
                }
            } catch (IOException ioe) {
                logger.log(
                    Level.WARNING,
                    "Unable close session " + session.id(),
                    ioe);
            }
        }

        stopped = true;
        logger.warning("PartitionWorker stopped " + partition.idString());
    }
}
