package ru.yandex.logbroker.topic;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.logging.Level;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logbroker.Stoppable;
import ru.yandex.logbroker.client.OffsetInfo;
import ru.yandex.logbroker.client.Partition;
import ru.yandex.logbroker.client.exception.LogbrockerClientException;
import ru.yandex.logbroker.client.exception.LogbrokerBadRequestException;
import ru.yandex.logbroker.client.exception.LogbrokerConflictException;
import ru.yandex.logbroker.config.ClientConfig;
import ru.yandex.logbroker.config.ImmutableTopicConfig;
import ru.yandex.logbroker.server.LogSkipUtil;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.stater.AbstractStatable;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class TopicConsumeManager
    extends AbstractStatable
    implements Runnable, Stoppable, GenericAutoCloseable<IOException>
{
    private static final int DEFAULT_DELAY = 1000;

    private final Map<String, WrappedWorker> workerMap = new HashMap<>();
    private final ExecutorService logbrokerConsumersExecutor;
    private final ImmutableTopicConfig context;
    private final ClientConfig clientConfig;
    private final PrefixedLogger logger;
    private final int regularWorkersLimit;
    private final int auxillaryWorkersLimit;
    private final int totalWorkersLimit;

    private volatile boolean stop = false;
    private volatile boolean stopped = false;
    private volatile boolean skipToEnd = false;
    private volatile Thread thread = null;

    private final ReadMetrics readMetrics;

    public TopicConsumeManager(
        final ImmutableTopicConfig config,
        final long metricsTimeFrame)
    {
        this.context = config;
        this.clientConfig = config.config().clientConfig();
        String prefix = config.logName() + '-' + config.dc();
        this.logger =
            context.logger().addPrefix("TopicManager-" + config.dc());

        this.regularWorkersLimit = clientConfig.workers();
        if (this.regularWorkersLimit < 0) {
            this.auxillaryWorkersLimit = 0;
        } else {
            this.auxillaryWorkersLimit = this.regularWorkersLimit;
        }

        this.totalWorkersLimit = regularWorkersLimit + auxillaryWorkersLimit;

        NamedThreadFactory threadFactory =
            new NamedThreadFactory(
                config.threadGroup(),
                prefix + "-worker-");

        if (totalWorkersLimit < 0) {
            this.logbrokerConsumersExecutor =
                Executors.newCachedThreadPool(threadFactory);
        } else {
            this.logbrokerConsumersExecutor =
                Executors.newFixedThreadPool(totalWorkersLimit, threadFactory);
        }

        this.logger.info("Init log consumer");
        this.logger.info("Topic " + context.topic());
        this.logger.info(
            "ClientId "
                + context.config().clientConfig().clientId());
        this.logger.info(
            "LB host " + context.config().hostConfig().host());
        this.logger.info("DC " + context.dc());
        TimeFrameQueue<Long> readChunks =
            new TimeFrameQueue<>(metricsTimeFrame);
        registerStater(
            new PassiveStaterAdapter<>(
                readChunks,
                new NamedStatsAggregatorFactory<>(
                    config.logName() + '-' + config.dc() + "-read-chunks_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        readMetrics = new ReadMetrics(readChunks);
    }

    public String dc() {
        return context.dc();
    }

    public String topicKey() {
        return context.logName();
    }

    public void skipToEnd() {
        this.skipToEnd = true;
    }

    public ImmutableTopicConfig config() {
        return this.context;
    }

    @Override
    public synchronized boolean stopped() {
        boolean result = stopped;
        for (WrappedWorker worker: workerMap.values()) {
            result &= worker.worker.stopped();
        }

        return result;
    }

    private PartitionWorker addWorker(
        final Partition partition,
        final boolean support)
    {
        PartitionWorker worker;
        if (support) {
            worker =
                new SupportPartitionWorker(context, partition, readMetrics);
        } else {
            worker = new PartitionWorker(context, partition, readMetrics);
        }

        Future<?> future = logbrokerConsumersExecutor.submit(worker);
        workerMap.put(partition.idString(), new WrappedWorker(future, worker));
        return worker;
    }

    private void process() throws InterruptedException {
        logger.info("Starting consuming topic " + context.topic());
        long timeout = 0;
        while (!stop) {
            if (timeout != 0) {
                Thread.sleep(timeout);
            } else {
                timeout = DEFAULT_DELAY;
            }

            int totalPartitions;
            List<OffsetInfo> offsets;

            // clean up workers
            boolean cleaned =
                workerMap.entrySet()
                    .removeIf(e -> e.getValue().future.isDone());
            if (cleaned) {
                context.logger().info("Completed workers cleaned");
            }

            if (workerMap.size() >= totalWorkersLimit) {
                timeout = clientConfig.sessionRetryDelay();
                continue;
            }

            try {
                offsets = this.context.client().offsets(context.topic());
                totalPartitions = offsets.size();
            } catch (LogbrockerClientException lce) {
                context.logger().log(
                    Level.WARNING,
                    "Failed to retrieve offsets for topic " + context.topic(),
                    lce);
                continue;
            }

            if (totalPartitions <= 0) {
                timeout = clientConfig.retryNoPartitionsDelay();
                context.logger().warning(
                    "No partitions detected, retry in " + timeout);
                continue;
            }

            List<Partition> partitions;

            try {
                partitions = this.context.client().suggest(
                    context.topic(),
                    totalPartitions * 2);
            } catch (LogbrokerBadRequestException lbe) {
                timeout = clientConfig.retryNoPartitionsDelay();
                context.logger().log(
                    Level.WARNING,
                    "bad request, retry in " + timeout + " ms",
                    lbe);
                continue;
            } catch (LogbrokerConflictException lce) {
                timeout = clientConfig.sessionRetryDelay();
                logger.info("Conflict, retry in " + timeout + " ms ");
                continue;
            } catch (LogbrockerClientException lce) {
                logger.log(
                    Level.WARNING,
                    "Failed to request partitions",
                    lce);
                timeout = clientConfig.networkRetryDelay();
                continue;
            }

            if (skipToEnd) {
                context.logger().warning("SkipToEnd activated, check workers");
                if (!workerMap.isEmpty()) {
                    context.logger().warning("Workers are alive, stopping");
                    workerMap.forEach((k, v) -> v.worker.stop());
                    timeout = clientConfig.sessionRetryDelay();
                    continue;
                }

                if (partitions.isEmpty()
                    || partitions.size() != offsets.size())
                {
                    continue;
                }

                new LogSkipUtil(context).skipLog(partitions, offsets);
                skipToEnd = false;
            }

            if (partitions.isEmpty()) {
                timeout = clientConfig.retryNoPartitionsDelay();
                context.logger().warning(
                    "No partitions in topic, retry in " + timeout);
                continue;
            }

            for (Partition partition: partitions) {
                if (!workerMap.containsKey(partition.idString())) {
                    if (workerMap.size() >= this.regularWorkersLimit) {
                        logger.info(
                            "Starting Support worker " + partition.toString());
                        addWorker(partition, true);
                    } else {
                        logger.info(
                            "Starting worker for partition "
                                + partition.toString());
                        addWorker(partition, false);
                    }
                }

                timeout = clientConfig.sessionRetryDelay();
                continue;
            }
        }
    }

    @Override
    public void run() {
        Thread.currentThread().setUncaughtExceptionHandler(
            new Thread.UncaughtExceptionHandler() {
                @Override
                public void uncaughtException(
                    final Thread t,
                    final Throwable e)
                {
                    logger.log(Level.SEVERE, "TopicConsumeManager died", e);
                    System.exit(1);
                }
            });

        try {
            process();

            logger.info("Stopping workers");
            logbrokerConsumersExecutor.shutdown();
            workerMap.forEach((k, v) -> v.worker.stop());
        } catch (InterruptedException ie) {
            context.logger().log(Level.WARNING, "Manager interrupted", ie);
        } catch (RejectedExecutionException ree) {
            context.logger().log(Level.WARNING, "Cannot run worker", ree);
        } finally {
            logbrokerConsumersExecutor.shutdownNow();
        }

        logger.warning("TopicConsumeManager stopped");
        stopped = true;
    }

    public void start() {
        logger.info(
            "Starting "
                + totalWorkersLimit
                + " consume workers, where "
                + auxillaryWorkersLimit + " are support workers");

        if (thread != null) {
            throw new RuntimeException("Manager already started");
        }

        thread = new Thread(
            config().threadGroup(),
            this);
        thread.start();
    }

    @Override
    public synchronized void stop() {
        logger.info("Request stopping workers");
        stop = true;
        logbrokerConsumersExecutor.shutdown();
        workerMap.values().forEach(w -> w.worker.stop());
    }

    @Override
    public void close() throws IOException {
        if (logbrokerConsumersExecutor != null) {
            logbrokerConsumersExecutor.shutdownNow();
        }

        if (thread != null) {
            thread.interrupt();
        }
    }

    private static final class WrappedWorker {
        private final Future<?> future;
        private final PartitionWorker worker;

        private WrappedWorker(
            final Future<?> future,
            final PartitionWorker worker)
        {
            this.future = future;
            this.worker = worker;
        }
    }
}
