package ru.yandex.logger;

import java.nio.charset.Charset;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import ru.yandex.parser.config.ConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractLoggerConfigBuilder
    <T extends AbstractLoggerConfigBuilder<T>>
    implements ConfigBuilder<T>, LoggerConfig
{
    private Map<String, LoggerFileConfigBuilder> files;
    private String separator;
    private Level logLevel;
    private Charset charset;

    protected AbstractLoggerConfigBuilder(final LoggerConfig config) {
        files(config.files());
        separator(config.separator());
        logLevel(config.logLevel());
        charset(config.charset());
    }

    protected AbstractLoggerConfigBuilder(
        final IniConfig config,
        final LoggerConfig defaults)
        throws ConfigException
    {
        this(config, defaults, defaults.files().get(LoggerConfig.DEFAULT));
    }

    protected AbstractLoggerConfigBuilder(
        final IniConfig config,
        final LoggerConfig defaults,
        final LoggerFileConfig defaultFileConfig)
        throws ConfigException
    {
        separator = config.getString("separator", defaults.separator());
        logLevel = config.getLogLevel("level.min", defaults.logLevel());
        charset = config.getCharset("charset", defaults.charset());

        Set<String> sections = new HashSet<>(config.sections().keySet());
        sections.remove("date");
        sections.remove("level");

        if (sections.isEmpty() || !config.keys().isEmpty()) {
            files =
                Collections.singletonMap(
                    LoggerConfig.DEFAULT,
                    new LoggerFileConfigBuilder(config, defaultFileConfig));
        } else {
            files = new LinkedHashMap<>();
            for (Map.Entry<String, IniConfig> entry
                : config.sections().entrySet())
            {
                LoggerFileConfig defaultSubFileConfig =
                    defaults.files().get(entry.getKey());
                if (defaultSubFileConfig == null) {
                    defaultSubFileConfig =
                        LoggerConfigDefaults.DEFAULT_CONFIGS.get(
                            entry.getKey());
                }

                if (defaultSubFileConfig == null) {
                    defaultSubFileConfig = defaultFileConfig;
                }

                files.put(
                    entry.getKey(),
                    new LoggerFileConfigBuilder(
                        entry.getValue(),
                        defaultSubFileConfig));
            }

            if (files.isEmpty()) {
                defaults.files().forEach((k, v) -> files.put(
                    k,
                    new LoggerFileConfigBuilder(v)));
            }
        }
    }

    @Override()
    public final Map<String, LoggerFileConfigBuilder> files() {
        return files;
    }

    public T files(final Map<String, ? extends LoggerFileConfig> value) {
        Map<String, LoggerFileConfigBuilder> files = new LinkedHashMap<>();
        for (Map.Entry<String, ? extends LoggerFileConfig> entry
            : value.entrySet())
        {
            files.put(
                entry.getKey(),
                new LoggerFileConfigBuilder(entry.getValue()));
        }

        this.files = files;
        return self();
    }

    @Override
    public String separator() {
        return separator;
    }

    public T separator(final String separator) {
        this.separator = separator;
        return self();
    }

    @Override
    public Level logLevel() {
        return logLevel;
    }

    public T logLevel(final Level logLevel) {
        this.logLevel = logLevel;
        return self();
    }

    @Override
    public Charset charset() {
        return charset;
    }

    public T charset(final Charset charset) {
        this.charset = charset;
        return self();
    }

    public LoggerFileConfigBuilder single() {
        if (files.size() == 0) {
            return null;
        }

        if (files.size() > 1) {
            throw new IllegalStateException("More than one file");
        }

        return files.entrySet().iterator().next().getValue();
    }

    public T add(final LoggerFileConfig config) {
        this.files.put(
            LoggerConfig.DEFAULT,
            new LoggerFileConfigBuilder(config));
        return self();
    }
}
