package ru.yandex.logger;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.ErrorManager;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import org.jctools.queues.MpscArrayQueue;

import ru.yandex.charset.Encoder;
import ru.yandex.function.OutputStreamProcessorAdapter;
import ru.yandex.function.StringBuilderVoidProcessor;
import ru.yandex.io.FsyncingFileOutputStream;
import ru.yandex.util.timesource.TimeSource;

public class AsyncStreamHandler extends StreamHandlerBase {
    public static final int QUEUE_SLEEP = 50;
    private static final int FILE_BUFFER_SIZE = 1024 * 1024;
    private static final int PROCESSOR_COUNT =
        Runtime.getRuntime().availableProcessors();
    private static final Level FAKE_LEVEL =
        new Level("", Integer.MAX_VALUE) {
            private static final long serialVersionUID = 0;
        };

    private static final LogRecord FLUSH_MSG = new LogRecord(FAKE_LEVEL, "");
    private static final LogRecord CLOSE_MSG = new LogRecord(FAKE_LEVEL, "");
    private static final Message FAKE_FLUSH_MSG = new Message(FLUSH_MSG) {
        @Override
        public void completed() {
        }
    };

    private final Lock lock;
    private final Condition queueNotFull;
    private final Condition queueNotEmpty;
    private final MpscArrayQueue<Message> logQueue;
    private final AsyncWriter writer;
    private final int notFullThreshold;
    private final int memoryLimit;
    private final int fsyncThreshold;
    private final int queueLength;
    private final File file;
    private final AtomicInteger memoryUsed = new AtomicInteger(0);
    private long prevOverrunWarning = 0;
    private volatile
        StringBuilderVoidProcessor<byte[], CharacterCodingException>
            encoder =
                new StringBuilderVoidProcessor<>(
                    new Encoder(Charset.defaultCharset()));

    public AsyncStreamHandler(final ImmutableLoggerFileConfig config)
        throws FileNotFoundException
    {
        file = config.file();
        memoryLimit = config.memoryLimit();
        fsyncThreshold = config.fsyncThreshold();
        queueLength = config.queueLength();
        notFullThreshold = Math.max(queueLength - PROCESSOR_COUNT, 0);
        logQueue = new MpscArrayQueue<>(queueLength);
        lock = new ReentrantLock();
        queueNotFull = lock.newCondition();
        queueNotEmpty = lock.newCondition();
        writer = new AsyncWriter(config.file());
        writer.setDaemon(true);
    }

    public AsyncStreamHandler start() {
        writer.start();
        Runtime.getRuntime().addShutdownHook(
            new Thread() {
                @Override
                public void run() {
                    flush();
                }
            });
        return this;
    }

    private void warnOverrun() {
        final long time = TimeSource.INSTANCE.currentTimeMillis();
        if (time - prevOverrunWarning
            > TimeUnit.SECONDS.toMillis(1))
        {
            StringBuilder sb = new StringBuilder(2 << (2 + 2));
            prevOverrunWarning = time;
            sb.append("Logger<");
            sb.append(file.getName());
            sb.append(">: memory/queue overrun: mem: ");
            sb.append(memoryUsed.get());
            sb.append('/');
            sb.append(memoryLimit);
            sb.append(" queue: ");
            sb.append(logQueue.size());
            sb.append('/');
            sb.append(notFullThreshold);
            System.err.println(new String(sb));
        }
    }

    private void putAndWait(final Message message, final int failureType) {
        try {
            queuePut(message);
            message.await();
        } catch (InterruptedException e) {
            reportError(null, e, failureType);
        }
    }

    private boolean memoryFull() {
        if (memoryLimit > 0) {
            return memoryUsed.get() >= memoryLimit;
        } else {
            return false;
        }
    }

    private void retainMemory(final Message message) {
        if (memoryLimit > 0) {
            memoryUsed.addAndGet(message.size());
        }
    }

    private void releaseMemory(final Message message) {
        if (memoryLimit > 0) {
            memoryUsed.addAndGet(-message.size());
        }
    }

    private void queuePut(final Message message)
        throws InterruptedException
    {
        while (memoryFull() || !logQueue.offer(message)) {
            lock.lock();
            try {
                warnOverrun();
                queueNotFull.await(QUEUE_SLEEP, TimeUnit.MILLISECONDS);
            } finally {
                lock.unlock();
            }
        }
        retainMemory(message);
        if (logQueue.size() == 1) {
            lock.lock();
            try {
                queueNotEmpty.signal();
            } finally {
                lock.unlock();
            }
        }
    }

    private Message queuePoll() {
        final boolean wasFull = memoryFull();
        Message m = logQueue.poll();
        if (m != null) {
            int queueLength = logQueue.size();
            releaseMemory(m);
            final boolean nowFull = memoryFull();
            if (queueLength == notFullThreshold
                || (wasFull && !nowFull))
            {
                lock.lock();
                try {
                    queueNotFull.signalAll();
                } finally {
                    lock.unlock();
                }
            }
        }
        return m;
    }

    private Message queueTake() throws InterruptedException {
        Message m = queuePoll();
        if (m != null) {
            return m;
        }
        lock.lock();
        try {
            while ((m = queuePoll()) == null) {
                queueNotEmpty.await(QUEUE_SLEEP, TimeUnit.MILLISECONDS);
            }
        } finally {
            lock.unlock();
        }
        return m;
    }

    protected void setOutputStream(final OutputStream out) {
        LogRecord record = new LogRecord(FAKE_LEVEL, "");
        record.setParameters(new Object[]{out});
        putAndWait(new Message(record), ErrorManager.GENERIC_FAILURE);
    }

    @Override
    @SuppressWarnings("UnsynchronizedOverridesSynchronized")
    public void setEncoding(final String encoding)
        throws UnsupportedEncodingException
    {
        super.setEncoding(encoding);
        if (encoding == null) {
            encoder =
                new StringBuilderVoidProcessor<>(
                    new Encoder(Charset.defaultCharset()));
        } else {
            try {
                encoder =
                    new StringBuilderVoidProcessor<>(
                        new Encoder(Charset.forName(encoding)));
            } catch (Exception e) {
                UnsupportedEncodingException ex =
                    new UnsupportedEncodingException(
                        "Unsupported encoding: " + encoding);
                ex.initCause(e);
                throw ex;
            }
        }
    }

    @Override
    public void close() {
        putAndWait(new Message(CLOSE_MSG), ErrorManager.CLOSE_FAILURE);
    }

    @Override
    public void flush() {
        putAndWait(new Message(FLUSH_MSG), ErrorManager.FLUSH_FAILURE);
    }

    @Override
    public void publish(final LogRecord record) {
        if (isLoggable(record)) {
            record.setResourceBundleName(Thread.currentThread().getName());
            try {
                queuePut(new Message(record));
            } catch (InterruptedException e) {
                reportError(null, e, ErrorManager.WRITE_FAILURE);
            }
        }
    }

    protected OutputStream createStream(final File file)
        throws FileNotFoundException
    {
        if (fsyncThreshold > 0) {
            return new BufferedOutputStream(
                new FsyncingFileOutputStream(file, true, fsyncThreshold),
                Math.min(FILE_BUFFER_SIZE, fsyncThreshold));
        } else {
            return new BufferedOutputStream(
                new FileOutputStream(file, true),
                FILE_BUFFER_SIZE);
        }
    }

    private class AsyncWriter extends Thread {
        private OutputStream out;
        private OutputStreamProcessorAdapter adapter;

        AsyncWriter(final File file) throws FileNotFoundException {
            super("AsyncLogWriter:" + file.getAbsolutePath());
            out = createStream(file);
            adapter = new OutputStreamProcessorAdapter(out);
        }

        private void flush() {
            try {
                out.flush();
            } catch (IOException e) {
                reportError(null, e, ErrorManager.FLUSH_FAILURE);
            }
        }

        private void close() {
            try {
                out.close();
            } catch (IOException e) {
                reportError(null, e, ErrorManager.CLOSE_FAILURE);
            }
        }

        @SuppressWarnings("ReferenceEquality")
        private void handleMessage(final Message msg)
            throws InterruptedException
        {
            LogRecord record = msg.record();
            if (record.getLevel() == FAKE_LEVEL) {
                if (record == FLUSH_MSG) {
                    flush();
                    msg.completed();
                } else if (record == CLOSE_MSG) {
                    close();
                    msg.completed();
                } else {
                    flush();
                    close();
                    out = (OutputStream) record.getParameters()[0];
                    adapter = new OutputStreamProcessorAdapter(out);
                    msg.completed();
                }
            } else {
                try {
                    encoder.process(formatter.formatSB(record));
                    encoder.processWith(adapter);
                } catch (IOException e) {
                    reportError(null, e, ErrorManager.WRITE_FAILURE);
                }
            }
        }

        @Override
        public void run() {
            while (true) {
                try {
                    Message msg = queuePoll();
                    if (msg == null) {
                        handleMessage(FAKE_FLUSH_MSG);
                        msg = queueTake();
                    }
                    handleMessage(msg);
                } catch (InterruptedException e) {
                    break;
                }
            }
        }
    }

    private static class Message {
        private final LogRecord record;
        private boolean incomplete = true;

        Message(final LogRecord record) {
            this.record = record;
        }

        public LogRecord record() {
            return record;
        }

        public synchronized void await() throws InterruptedException {
            while (incomplete) {
                wait();
            }
        }

        public synchronized void completed() {
            incomplete = false;
            notifyAll();
        }

        private int size() {
            int size;
            if (record == null || record.getMessage() == null) {
                size = 0;
            } else {
                size = record.getMessage().length() << 1;
            }
            return size;
        }
    }
}
