package ru.yandex.logger;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.ConsoleHandler;
import java.util.logging.Handler;

import ru.yandex.parser.config.ConfigException;

public class HandlersManager {
    private final Map<ImmutableLoggerFileConfig, Handler> consoleHandlers =
        new HashMap<>();
    private final Map<File, Entry> files = new HashMap<>();

    public Handler handler(
        final ImmutableLoggerConfig loggerConfig,
        final ImmutableLoggerFileConfig config)
        throws ConfigException
    {
        File file = config.file();
        Handler handler;
        if (file == null) {
            handler = consoleHandlers.get(config);
            if (handler == null) {
                handler = createHandler(loggerConfig, config);
                consoleHandlers.put(config, handler);
            }
        } else {
            Entry entry = files.get(file);
            if (entry == null) {
                handler = createHandler(loggerConfig, config);
                files.put(file, new Entry(handler, config));
            } else if (entry.config().equals(config)) {
                handler = entry.handler();
            } else {
                throw new ConfigException(
                    "Config mismatch for " + file + ", requested: " + config
                    + ", but previous was: " + entry.config());
            }
        }
        return handler;
    }

    private Handler createHandler(
        final ImmutableLoggerConfig loggerConfig,
        final ImmutableLoggerFileConfig fileConfig)
        throws ConfigException
    {
        Handler handler;
        if (fileConfig.file() == null) {
            handler = new ConsoleHandler();
        } else {
            try {
                handler = fileConfig.rotate().createHandler(fileConfig);
            } catch (FileNotFoundException e) {
                throw new ConfigException("Bad log file set", e);
            }
        }

        try {
            handler.setEncoding(loggerConfig.charset().name());
        } catch (UnsupportedEncodingException e) {
            throw new ConfigException("Bad charset set for log", e);
        }

        SBFormatter formatter;
        if (fileConfig.queueLength() == 0) {
            formatter = new CustomFormatter(loggerConfig, fileConfig);
        } else {
            formatter = new AsyncCustomFormatter(loggerConfig, fileConfig);
        }
        handler.setFormatter(formatter);
        handler.setLevel(loggerConfig.logLevel());
        return handler;
    }

    public Set<Handler> handlers() {
        Set<Handler> handlers = new HashSet<>(consoleHandlers.values());
        for (Entry entry: files.values()) {
            handlers.add(entry.handler());
        }
        return handlers;
    }

    private static class Entry {
        private final Handler handler;
        private final ImmutableLoggerFileConfig config;

        Entry(
            final Handler handler,
            final ImmutableLoggerFileConfig config)
        {
            this.handler = handler;
            this.config = config;
        }

        public Handler handler() {
            return handler;
        }

        public ImmutableLoggerFileConfig config() {
            return config;
        }
    }
}

