package ru.yandex.logger;

import java.util.concurrent.atomic.AtomicLong;

import ru.yandex.util.timesource.TimeSource;

public class IdGenerator {
    private static final long MILLIS_MULTIPLIER = 10L;
    private static final int DEFAULT_LENGTH = 6;
    private static final int DEFAULT_RADIX = 36;
    private static final char[] ENC_TABLE = new char[] {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
        'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
        'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd',
        'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
        'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x',
        'y', 'z'
    };
    public static final int MAX_RADIX = ENC_TABLE.length;

    private final AtomicLong count;
    private final int length;
    private final int radix;
    private final long mask;

    public IdGenerator() {
        this(defaultStart());
    }

    public IdGenerator(final long start) {
        this(start, DEFAULT_LENGTH, DEFAULT_RADIX);
    }

    public IdGenerator(final int length, final int radix) {
        this(defaultStart(), length, radix);
    }

    public IdGenerator(final long start, final int length, final int radix) {
        if (length <= 0) {
            throw new IllegalArgumentException("Bad length: " + length);
        }

        if (radix <= 0 || radix > MAX_RADIX) {
            throw new IllegalArgumentException("Bad radix: " + radix);
        }

        count = new AtomicLong(start);
        this.length = length;
        this.radix = radix;
        long mask = radix;
        for (int i = 1; i < length; ++i) {
            mask *= radix;
        }
        this.mask = mask;
    }

    private static long defaultStart() {
        return TimeSource.INSTANCE.currentTimeMillis() * MILLIS_MULTIPLIER;
    }

    public String next() {
        char[] cbuf = new char[length];
        long id = count.getAndIncrement() % mask;
        if (id < 0) {
            id += mask;
        }
        for (int i = 1; i <= length; ++i, id /= radix) {
            cbuf[length - i] = ENC_TABLE[(int) (id % radix)];
        }
        return new String(cbuf);
    }
}

