package ru.yandex.logger;

import java.nio.charset.Charset;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.ImmutableConfig;

public class ImmutableLoggerConfig
    implements ImmutableConfig, LoggerConfig
{
    private static final int HASH_CODE_CONST = 31;
    private final Map<String, ImmutableLoggerFileConfig> files;
    private final String separator;
    private final Level logLevel;
    private final Charset charset;

    public ImmutableLoggerConfig(final LoggerConfig config)
        throws ConfigException
    {
        Map<String, ImmutableLoggerFileConfig> files = new LinkedHashMap<>();
        for (Map.Entry<String, ? extends LoggerFileConfig> entry
            : config.files().entrySet())
        {
            files.put(
                entry.getKey(),
                new ImmutableLoggerFileConfig(entry.getValue()));
        }

        this.files = Collections.unmodifiableMap(files);

        this.separator = config.separator();
        this.logLevel = config.logLevel();
        this.charset = config.charset();
    }

    @Override
    public Map<String, ImmutableLoggerFileConfig> files() {
        return files;
    }

    public ImmutableLoggerFileConfig single() {
        if (files.size() == 0) {
            return null;
        }

        if (files.size() > 1) {
            throw new IllegalStateException("More than one file");
        }

        return files.entrySet().iterator().next().getValue();
    }

    public Logger build(final HandlersManager handlersManager)
        throws ConfigException
    {
        Logger logger = Logger.getAnonymousLogger();
        for (ImmutableLoggerFileConfig config: files().values()) {
            logger.addHandler(handlersManager.handler(this, config));
        }

        logger.setLevel(logLevel());
        logger.setUseParentHandlers(false);
        return logger;
    }

    public PrefixedLogger buildPrefixed(final HandlersManager handlersManager)
        throws ConfigException
    {
        return new PrefixedLogger(build(handlersManager), "", separator);
    }

    public static ImmutableLoggerConfig create(final LoggerConfig config)
        throws ConfigException
    {
        if (config == null) {
            return null;
        } else {
            return new ImmutableLoggerConfig(config);
        }
    }

    @Override
    public String separator() {
        return separator;
    }

    @Override
    public Level logLevel() {
        return logLevel;
    }

    @Override
    public Charset charset() {
        return charset;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ImmutableLoggerConfig)) {
            return false;
        }

        ImmutableLoggerConfig that = (ImmutableLoggerConfig) o;

        if (!files.equals(that.files)) {
            return false;
        }
        if (!separator.equals(that.separator)) {
            return false;
        }
        if (!logLevel.equals(that.logLevel)) {
            return false;
        }
        return charset.equals(that.charset);
    }

    @Override
    public int hashCode() {
        int result = files.hashCode();
        result = HASH_CODE_CONST * result + separator.hashCode();
        result = HASH_CODE_CONST * result + logLevel.hashCode();
        result = HASH_CODE_CONST * result + charset.hashCode();
        return result;
    }
}
