package ru.yandex.logger;

import java.util.logging.Logger;

import ru.yandex.collection.PatternMap;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.parser.config.ConfigException;

public class ImmutableLoggersConfig implements LoggersConfig {
    private final HandlersManager handlersManager;
    private final ImmutableLoggerConfig stdoutConfig;
    private final ImmutableLoggerConfig stderrConfig;
    private final PatternMap<RequestInfo, ImmutableLoggerConfig> loggersConfigs;
    private final PatternMap<RequestInfo, ImmutableLoggerConfig>
        accessLoggersConfigs;
    private final Logger stdout;
    private final Logger stderr;
    private final PatternMap<RequestInfo, PrefixedLogger> loggers;
    private final PatternMap<RequestInfo, Logger> accessLoggers;

    public ImmutableLoggersConfig(final LoggersConfig config)
        throws ConfigException
    {
        this(config, new HandlersManager());
    }

    public ImmutableLoggersConfig(
        final LoggersConfig config,
        final HandlersManager handlersManager)
        throws ConfigException
    {
        this.handlersManager = handlersManager;
        stdoutConfig = ImmutableLoggerConfig.create(config.stdout());
        stderrConfig = ImmutableLoggerConfig.create(config.stderr());
        loggersConfigs =
            config.loggers().transform(ImmutableLoggerConfig::new);
        accessLoggersConfigs =
            config.accessLoggers().transform(ImmutableLoggerConfig::new);
        if (stdoutConfig == null) {
            stdout = null;
        } else {
            stdout = stdoutConfig.build(handlersManager);
        }
        if (stderrConfig == null) {
            stderr = null;
        } else {
            stderr = stderrConfig.build(handlersManager);
        }
        accessLoggers = accessLoggersConfigs.transform(
            cfg -> cfg.buildPrefixed(handlersManager));
        loggers = loggersConfigs.transform(
            cfg -> cfg.buildPrefixed(handlersManager));
    }

    public HandlersManager handlersManager() {
        return handlersManager;
    }

    @Override
    public ImmutableLoggerConfig stdout() {
        return stdoutConfig;
    }

    @Override
    public ImmutableLoggerConfig stderr() {
        return stderrConfig;
    }

    @Override
    public PatternMap<RequestInfo, ImmutableLoggerConfig> loggers() {
        return loggersConfigs;
    }

    @Override
    public PatternMap<RequestInfo, ImmutableLoggerConfig> accessLoggers() {
        return accessLoggersConfigs;
    }

    public Logger loggerForStdout() {
        return stdout;
    }

    public Logger loggerForStderr() {
        return stderr;
    }

    public PatternMap<RequestInfo, PrefixedLogger> preparedLoggers() {
        return loggers;
    }

    public PatternMap<RequestInfo, Logger> preparedAccessLoggers() {
        return accessLoggers;
    }
}

