package ru.yandex.logger;

import java.io.File;
import java.util.Locale;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.IntegerMemorySizeParser;
import ru.yandex.parser.string.NonNegativeIntegerValidator;

public class LoggerFileConfigBuilder implements LoggerFileConfig {
    private File file;
    private FileRotateType rotate;
    private FormatParser outputFormat;
    private String dateFormat;
    private Locale dateLocale;
    private String logFormat;
    private int queueLength;
    private int memoryLimit;
    private int fsyncThreshold;

    public LoggerFileConfigBuilder() {
        this(LoggerFileConfigDefaults.INSTANCE);
    }

    public LoggerFileConfigBuilder(final LoggerFileConfig config) {
        file = config.file();
        rotate = config.rotate();
        dateFormat = config.dateFormat();
        dateLocale = config.dateLocale();
        logFormat = config.logFormat();
        queueLength = config.queueLength();
        memoryLimit = config.memoryLimit();
        fsyncThreshold = config.fsyncThreshold();
        outputFormat = config.outputFormat();
    }

    public LoggerFileConfigBuilder(
        final IniConfig config)
        throws ConfigException
    {
        this(config, LoggerFileConfigDefaults.INSTANCE);
    }

    public LoggerFileConfigBuilder(
        final IniConfig config,
        final LoggerFileConfig defaults)
        throws ConfigException
    {
        file = config.getOutputFile("file", defaults.file());
        if (file == null) {
            rotate = FileRotateType.NONE;
            queueLength = 0;
            memoryLimit = 0;
        } else {
            rotate = config.getEnum(
                FileRotateType.class,
                "rotate",
                defaults.rotate());
            queueLength = config.get(
                "buffer",
                defaults.queueLength(),
                NonNegativeIntegerValidator.INSTANCE);
            memoryLimit = config.get(
                "memory-limit",
                defaults.memoryLimit(),
                IntegerMemorySizeParser.INSTANCE);
            fsyncThreshold = config.get(
                "fsync",
                defaults.memoryLimit(),
                IntegerMemorySizeParser.INSTANCE);
        }
        outputFormat = config.getEnum(
            FormatParser.class,
            "output-format",
            defaults.outputFormat());

        dateFormat = config.getString("date.format", defaults.dateFormat());
        dateLocale = config.getLocale("date.locale", defaults.dateLocale());
        String logFormat = config.getString("format", null);
        if (logFormat == null) {
            this.logFormat = defaults.logFormat();
        } else {
            this.logFormat = logFormat.replace("%{default}", defaults.logFormat());
        }
    }

    @Override
    public File file() {
        return file;
    }

    public LoggerFileConfigBuilder file(final File file) {
        this.file = file;
        return this;
    }

    @Override
    public FileRotateType rotate() {
        return rotate;
    }

    public LoggerFileConfigBuilder rotate(final FileRotateType rotate) {
        this.rotate = rotate;
        return this;
    }

    @Override
    public String dateFormat() {
        return dateFormat;
    }

    public LoggerFileConfigBuilder dateFormat(final String dateFormat) {
        this.dateFormat = dateFormat;
        return this;
    }

    @Override
    public Locale dateLocale() {
        return dateLocale;
    }

    public LoggerFileConfigBuilder dateLocale(
        final Locale dateLocale)
    {
        this.dateLocale = dateLocale;
        return this;
    }

    @Override
    public String logFormat() {
        return logFormat;
    }

    public LoggerFileConfigBuilder logFormat(final String logFormat) {
        this.logFormat = logFormat;
        return this;
    }

    @Override
    public int queueLength() {
        return queueLength;
    }

    public LoggerFileConfigBuilder queueLength(final int queueLength) {
        this.queueLength = queueLength;
        return this;
    }

    @Override
    public int memoryLimit() {
        return memoryLimit;
    }

    public LoggerFileConfigBuilder memoryLimit(final int memoryLimit) {
        this.memoryLimit = memoryLimit;
        return this;
    }

    @Override
    public int fsyncThreshold() {
        return fsyncThreshold;
    }

    public LoggerFileConfigBuilder fsyncThreshold(final int fsyncThreshold) {
        this.fsyncThreshold = fsyncThreshold;
        return this;
    }

    @Override
    public FormatParser outputFormat() {
        return outputFormat;
    }

    public LoggerFileConfigBuilder outputFormat(
        final FormatParser outputFormat)
    {
        this.outputFormat = outputFormat;
        return this;
    }

    public ImmutableLoggerFileConfig build() throws ConfigException {
        return new ImmutableLoggerFileConfig(this);
    }
}

