package ru.yandex.logger;

import java.util.Map;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.collection.PatternStringPredicate;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.request.RequestPatternParser;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class LoggersConfigBuilder implements LoggersConfig {
    public static final String STDOUT = "stdout";
    public static final String STDERR = "stderr";
    public static final String LOG = "log";
    public static final String ACCESSLOG = "accesslog";

    private LoggerConfigBuilder stdout;
    private LoggerConfigBuilder stderr;
    private PatternMap<RequestInfo, LoggerConfigBuilder> loggers;
    private PatternMap<RequestInfo, LoggerConfigBuilder> accessLoggers;

    public LoggersConfigBuilder() {
        this(LoggersConfigDefaults.INSTANCE);
    }

    public LoggersConfigBuilder(final LoggersConfig config) {
        stdout(config.stdout());
        stderr(config.stderr());
        loggers(config.loggers());
        accessLoggers(config.accessLoggers());
    }

    public LoggersConfigBuilder(final IniConfig config)
        throws ConfigException
    {
        this(config, LoggersConfigDefaults.INSTANCE);
    }

    public LoggersConfigBuilder(
        final IniConfig config,
        final LoggersConfig defaults)
        throws ConfigException
    {
        stdout = createStdoutConfigBuilder(
            config.sectionOrNull(STDOUT),
            defaults.stdout());
        stderr = createStdoutConfigBuilder(
            config.sectionOrNull(STDERR),
            defaults.stderr());
        loggers = loadLoggers(config.section(LOG), defaults.loggers());
        accessLoggers =
            loadLoggers(config.section(ACCESSLOG), defaults.accessLoggers());
    }

    private static LoggerConfigBuilder createStdoutConfigBuilder(
        final IniConfig config,
        final LoggerConfig defaults)
        throws ConfigException
    {
        LoggerConfigBuilder builder;
        if (config == null) {
            if (defaults == null) {
                builder = null;
            } else {
                builder = new LoggerConfigBuilder(defaults);
            }
        } else {
            if (defaults == null) {
                builder = new LoggerConfigBuilder(
                    config,
                    StdoutLoggerConfigDefaults.INSTANCE);
            } else {
                builder = new LoggerConfigBuilder(config, defaults);
            }
        }
        return builder;
    }

    public static PatternMap<RequestInfo, LoggerConfigBuilder> loadLoggers(
        final IniConfig section,
        final PatternMap<RequestInfo, ? extends LoggerConfig> defaults)
        throws ConfigException
    {
        PatternMap<RequestInfo, LoggerConfigBuilder> loggers =
            new PatternMap<>(
                new LoggerConfigBuilder(section, defaults.asterisk()));
        for (Map.Entry<String, IniConfig> entry
                : section.sections().entrySet())
        {
            String subsection = entry.getKey();
            if (PatternStringPredicate.INSTANCE.test(subsection)) {
                Pattern<RequestInfo> pattern =
                    RequestPatternParser.INSTANCE.apply(subsection);
                loggers.put(
                    pattern,
                    new LoggerConfigBuilder(
                        entry.getValue(),
                        defaults.get(pattern)));
            }
        }
        return loggers;
    }

    private static LoggerConfigBuilder createConfigBuilder(
        final LoggerConfig config)
    {
        if (config == null) {
            return null;
        } else {
            return new LoggerConfigBuilder(config);
        }
    }

    @Override
    public LoggerConfigBuilder stdout() {
        return stdout;
    }

    public LoggersConfigBuilder stdout(final LoggerConfig stdout) {
        this.stdout = createConfigBuilder(stdout);
        return this;
    }

    @Override
    public LoggerConfigBuilder stderr() {
        return stderr;
    }

    public LoggersConfigBuilder stderr(final LoggerConfig stderr) {
        this.stderr = createConfigBuilder(stderr);
        return this;
    }

    @Override
    public PatternMap<RequestInfo, LoggerConfigBuilder> loggers() {
        return loggers;
    }

    public LoggersConfigBuilder loggers(
        final PatternMap<RequestInfo, ? extends LoggerConfig> loggers)
    {
        this.loggers =
            loggers.transform(config -> new LoggerConfigBuilder(config));
        return this;
    }

    @Override
    public PatternMap<RequestInfo, LoggerConfigBuilder> accessLoggers() {
        return accessLoggers;
    }

    public LoggersConfigBuilder accessLoggers(
        final PatternMap<RequestInfo, ? extends LoggerConfig> accessLoggers)
    {
        this.accessLoggers =
            accessLoggers.transform(config -> new LoggerConfigBuilder(config));
        return this;
    }

    public ImmutableLoggersConfig build() throws ConfigException {
        return new ImmutableLoggersConfig(this);
    }
}

