package ru.yandex.logger;

import java.util.ArrayList;
import java.util.List;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.parser.config.ConfigException;

public class PlainFormatContextBuilder implements FormatterContextBuilder {
    private final DateTimeFormatter dateFormatter;
    private final LogAppender separator;
    private final ThreadAppenderFactory threadAppenderFactory;
    private final List<LogAppender> appenders = new ArrayList<>();

    public PlainFormatContextBuilder(
        final ImmutableLoggerConfig loggerConfig,
        final ImmutableLoggerFileConfig fileConfig,
        final ThreadAppenderFactory appenderFactory)
        throws ConfigException
    {
        try {
            dateFormatter = DateTimeFormat
                .forPattern(fileConfig.dateFormat())
                .withLocale(fileConfig.dateLocale());
        } catch (Exception e) {
            throw new ConfigException("Bad date format specified", e);
        }

        if (loggerConfig.separator().length() == 1) {
            separator = new CharAppender(loggerConfig.separator().charAt(0));
        } else {
            separator = new StringAppender(loggerConfig.separator());
        }

        this.threadAppenderFactory = appenderFactory;
    }

    @Override
    public void visit(final char c) {
        appenders.add(new CharAppender(c));
    }

    @Override
    public void visitString(final String s) {
        appenders.add(new StringAppender(s));
    }

    @Override
    public void visitPlaceholder(final String placeholder) {
        switch (placeholder) {
            case "date":
                appenders.add(new DateAppender(dateFormatter));
                break;

            case "level":
                appenders.add(new LevelAppender());
                break;

            case "message":
                appenders.add(new MessageAppender());
                break;

            case "separator":
                appenders.add(separator);
                break;

            case "thread":
                appenders.add(threadAppenderFactory.create());
                break;

            default:
                appenders.add(new ParamAppender(placeholder));
                break;
        }
    }

    @Override
    public List<LogAppender> build() {
        return new ArrayList<>(appenders);
    }
}
