package ru.yandex.logger;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.ErrorManager;
import java.util.logging.LogRecord;

import ru.yandex.charset.Encoder;
import ru.yandex.function.OutputStreamProcessorAdapter;
import ru.yandex.function.StringBuilderVoidProcessor;

public class StreamHandler extends StreamHandlerBase {
    @SuppressWarnings("ThreadLocalUsage")
    private volatile ThreadLocalStringBuilderEncoder encoder =
        new ThreadLocalStringBuilderEncoder(Charset.defaultCharset());
    private final AtomicInteger concurrentWrites = new AtomicInteger();
    private volatile OutputStream out;
    private volatile OutputStreamProcessorAdapter adapter;

    public StreamHandler(final OutputStream out) {
        this.out = out;
        adapter = new OutputStreamProcessorAdapter(out);
    }

    protected synchronized void setOutputStream(final OutputStream out) {
        flush();
        close();
        this.out = out;
        adapter = new OutputStreamProcessorAdapter(out);
    }

    @Override
    @SuppressWarnings("UnsynchronizedOverridesSynchronized")
    public void setEncoding(final String encoding)
        throws UnsupportedEncodingException
    {
        super.setEncoding(encoding);
        if (encoding == null) {
            encoder =
                new ThreadLocalStringBuilderEncoder(Charset.defaultCharset());
        } else {
            try {
                encoder =
                    new ThreadLocalStringBuilderEncoder(
                        Charset.forName(encoding));
            } catch (Exception e) {
                UnsupportedEncodingException ex =
                    new UnsupportedEncodingException(
                        "Unsupported encoding: " + encoding);
                ex.initCause(e);
                throw ex;
            }
        }
    }

    @Override
    public synchronized void close() {
        try {
            out.close();
        } catch (IOException e) {
            reportError(null, e, ErrorManager.CLOSE_FAILURE);
        }
    }

    @Override
    public synchronized void flush() {
        try {
            out.flush();
        } catch (IOException e) {
            reportError(null, e, ErrorManager.FLUSH_FAILURE);
        }
    }

    @Override
    public void publish(final LogRecord record) {
        if (isLoggable(record)) {
            concurrentWrites.incrementAndGet();
            final boolean flush;
            try {
                StringBuilderVoidProcessor<byte[], CharacterCodingException>
                    encoder = this.encoder.get();
                try {
                    encoder.process(formatter.formatSB(record));
                    synchronized (this) {
                        encoder.processWith(adapter);
                    }
                } catch (CharacterCodingException e) {
                    reportError(null, e, ErrorManager.FORMAT_FAILURE);
                } catch (IOException e) {
                    reportError(null, e, ErrorManager.WRITE_FAILURE);
                }
            } finally {
                flush = concurrentWrites.decrementAndGet() == 0;
            }
            if (flush) {
                flush();
            }
        }
    }

    private static class ThreadLocalStringBuilderEncoder
        extends ThreadLocal<
            StringBuilderVoidProcessor<byte[], CharacterCodingException>>
    {
        private final Charset charset;

        ThreadLocalStringBuilderEncoder(final Charset charset) {
            this.charset = charset;
        }

        @Override
        protected StringBuilderVoidProcessor<byte[], CharacterCodingException>
            initialValue()
        {
            return new StringBuilderVoidProcessor<>(new Encoder(charset));
        }
    }
}

