package ru.yandex.logger.json;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.logger.FormatterContextBuilder;
import ru.yandex.logger.ImmutableLoggerConfig;
import ru.yandex.logger.ImmutableLoggerFileConfig;
import ru.yandex.logger.LogAppender;
import ru.yandex.logger.ThreadAppenderFactory;
import ru.yandex.parser.config.ConfigException;

public class JsonFormatContextBuilder implements FormatterContextBuilder {
    private final List<JsonLogAppender> appenders;
    private final Set<String> params;
    private final Map<String, String> envs;
    private final DateTimeFormatter dateFormatter;
    private final ImmutableLoggerFileConfig fileConfig;

    public JsonFormatContextBuilder(
        final ImmutableLoggerConfig loggerConfig,
        final ImmutableLoggerFileConfig fileConfig,
        final ThreadAppenderFactory appenderFactory)
        throws ConfigException
    {
        this.appenders = new ArrayList<>();
        this.params = new LinkedHashSet<>();
        this.envs = new LinkedHashMap<>();
        this.fileConfig = fileConfig;

        try {
            dateFormatter = DateTimeFormat
                .forPattern(fileConfig.dateFormat())
                .withLocale(fileConfig.dateLocale());
        } catch (Exception e) {
            throw new ConfigException("Bad date format specified", e);
        }
    }

    @Override
    public void visit(final char c) {
        if (!Character.isSpaceChar(c)) {
            throw new UnsupportedOperationException(
                "Chars are not supported in json format " + c + ' '
                    + fileConfig.logFormat());
        }
    }

    @Override
    public void visitString(final String s) {
        throw new UnsupportedOperationException(
            "Strings are not supported in json format " + s);
    }

    @SuppressWarnings("StringSplitter")
    private void parseOtherPlaceholder(final String placeholder) {
        if (placeholder.startsWith("env.")) {
            String[] split = placeholder.split("\\.");
            if (split.length == 2) {
                envs.put(split[1], split[1]);
            } else {
                envs.put(split[2], split[1]);
            }
        } else {
            params.add(placeholder);
        }
    }

    @Override
    public void visitPlaceholder(final String placeholder) {
        JsonLogFields field = null;
        try {
            field = JsonLogFields.valueOf(
                placeholder.toUpperCase(Locale.ENGLISH));
        } catch (IllegalArgumentException iae) {
            parseOtherPlaceholder(placeholder);
            return;
        }

        switch (field) {
            case MESSAGE:
                appenders.add(new MessageJsonLogAppender());
                break;
            case DATE:
                appenders.add(new DateJsonLogAppender(dateFormatter));
                break;
            case TIMESTAMP:
                appenders.add(new TimeJsonLogAppender());
                break;
            default:
                break;
        }
    }

    @Override
    public List<LogAppender> build() {
        if (!params.isEmpty()) {
            appenders.add(new JsonParamsLogAppender(params));
        }

        if (!envs.isEmpty()) {
            appenders.add(new JsonEnvsLogAppender(envs));
        }

        return Collections.singletonList(new JsonWriterAppender(appenders));
    }
}
