package ru.yandex.mulcagate;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;

public class MulcagateClient extends AsyncClient {
    private static final String BASE_URI = "/gate/dist-info/";

    private final HttpHost host;

    public MulcagateClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpHostConfig mulcagateConfig)
    {
        super(reactor, mulcagateConfig);
        host = mulcagateConfig.host();
    }

    public Future<List<StorageHost>> resolvePrimaryStidHosts(
        final String stid,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super List<StorageHost>> callback)
    {
        return execute(
            host,
            new BasicAsyncRequestProducerGenerator(
                BASE_URI + stid + "?primary-only"),
            PrimaryDistInfoConsumerFactory.OK,
            contextGenerator,
            callback);
    }

    public List<StorageHost> resolvePrimaryStidHostsSyncronously(
        final String stid,
        final RequestsListener listener,
        final Logger logger)
        throws HttpException
    {
        try {
            return resolvePrimaryStidHosts(
                stid,
                listener.createContextGeneratorFor(this),
                EmptyFutureCallback.INSTANCE).get();
        } catch (InterruptedException e) {
            throw new ServiceUnavailableException(e);
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            String message =
                "Failed to resolve hosts for stid " + stid
                + ':' + ' ' + listener.details();
            logger.log(Level.WARNING, message, cause);
            if (cause instanceof HttpException) {
                throw (HttpException) cause;
            } else {
                throw new BadGatewayException(message, cause);
            }
        }
    }

    public static List<HttpHost> toHttpHosts(
        final List<StorageHost> hosts,
        final int port)
    {
        List<HttpHost> result = new ArrayList<>(hosts.size());
        for (StorageHost host: hosts) {
            result.add(new HttpHost(host.hostname(), port));
        }
        return result;
    }
}

