package ru.yandex.mulcagate;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.function.StringProcessor;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;

public class StorageHost implements Comparable<StorageHost> {
    private static final int HOSTNAME_INDEX = 0;
    private static final int DC_INDEX = 1;
    private static final int ROLE_INDEX = 2;
    private static final int FASTBONE_HOSTNAME_INDEX = 3;
    private static final CollectionParser<String, List<String>, Exception>
        TSV_PARSER = new CollectionParser<>(
            NonEmptyValidator.INSTANCE,
            ArrayList::new,
            '\t');

    private final String hostname;
    private final String fastboneHostname;
    private final String dc;
    private final boolean primary;

    public StorageHost(final String str)
        throws MulcagateMalformedHostException
    {
        List<String> parts;
        try {
            parts = new StringProcessor<>(TSV_PARSER).process(str);
        } catch (Exception e) {
            throw new MulcagateMalformedHostException(
                "Failed to split: '" + str + '\'',
                e);
        }
        int size = parts.size();
        if (size != FASTBONE_HOSTNAME_INDEX
            && size != FASTBONE_HOSTNAME_INDEX + 1)
        {
            throw new MulcagateMalformedHostException(
                "Expected 3 or 4 tab separated values: '" + str + '\'');
        }
        hostname = parts.get(HOSTNAME_INDEX);
        dc = parts.get(DC_INDEX);
        primary = "primary".equals(parts.get(ROLE_INDEX));
        if (size == FASTBONE_HOSTNAME_INDEX) {
            fastboneHostname = hostname;
        } else {
            fastboneHostname = parts.get(FASTBONE_HOSTNAME_INDEX);
        }
    }

    public String hostname() {
        return hostname;
    }

    public String fastboneHostname() {
        return fastboneHostname;
    }

    public String dc() {
        return dc;
    }

    public boolean primary() {
        return primary;
    }

    @Override
    public int compareTo(final StorageHost other) {
        int cmp = hostname.compareTo(other.hostname);
        if (cmp == 0) {
            cmp = fastboneHostname.compareTo(other.fastboneHostname);
            if (cmp == 0) {
                cmp = dc.compareTo(other.dc);
                if (cmp == 0) {
                    cmp = Boolean.compare(primary, other.primary);
                }
            }
        }
        return cmp;
    }

    @Override
    public int hashCode() {
        int hashCode = hostname.hashCode() ^ dc.hashCode();
        if (!fastboneHostname.equals(hostname)) {
            hashCode ^= fastboneHostname.hashCode();
        }
        if (primary) {
            hashCode = ~hashCode;
        }
        return hashCode;
    }

    @Override
    public boolean equals(final Object o) {
        if (o instanceof StorageHost) {
            StorageHost other = (StorageHost) o;
            return primary == other.primary
                && hostname.equals(other.hostname)
                && fastboneHostname.equals(other.fastboneHostname)
                && dc.equals(other.dc);
        }
        return false;
    }

    @Override
    public String toString() {
        return hostname + ',' + fastboneHostname + ',' + dc + ',' + primary;
    }
}

