/*
  Copyright 2012 Stefano Chizzolini. http://www.pdfclown.org

  Contributors:
    * Stefano Chizzolini (original code developer, http://www.stefanochizzolini.it)

  This file should be part of the source code distribution of "PDF Clown library"
  (the Program): see the accompanying README files for more info.

  This Program is free software; you can redistribute it and/or modify it under the terms
  of the GNU Lesser General Public License as published by the Free Software Foundation;
  either version 3 of the License, or (at your option) any later version.

  This Program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY,
  either expressed or implied; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the License for more details.

  You should have received a copy of the GNU Lesser General Public License along with this
  Program (see README files); if not, go to the GNU website (http://www.gnu.org/licenses/).

  Redistribution and use, with or without modification, are permitted provided that such
  redistributions retain the above copyright notice, license and disclaimer, along with
  this list of conditions.
*/

package org.pdfclown.documents.interaction.navigation.page;

import org.pdfclown.PDF;
import org.pdfclown.VersionEnum;
import org.pdfclown.documents.Document;
import org.pdfclown.documents.interchange.metadata.Information;
import org.pdfclown.objects.PdfDictionary;
import org.pdfclown.objects.PdfDirectObject;
import org.pdfclown.objects.PdfName;
import org.pdfclown.objects.PdfObjectWrapper;

/**
  Article thread [PDF:1.7:8.3.2].

  @author Stefano Chizzolini (http://www.stefanochizzolini.it)
  @since 0.1.2
  @version 0.1.2, 12/28/12
*/
@PDF(VersionEnum.PDF11)
public class Article
  extends PdfObjectWrapper<PdfDictionary>
{
  // <class>
  // <static>
  // <interface>
  // <public>
  public static Article wrap(
    PdfDirectObject baseObject
    )
  {return baseObject != null ? new Article(baseObject) : null;}
  // </public>
  // </interface>
  // </static>

  // <dynamic>
  // <constructors>
  public Article(
    Document context
    )
  {
    super(
      context,
      new PdfDictionary(
        new PdfName[]
        {PdfName.Type},
        new PdfDirectObject[]
        {PdfName.Thread}
        )
      );
    context.getArticles().add(this);
  }

  private Article(
    PdfDirectObject baseObject
    )
  {super(baseObject);}
  // </constructors>

  // <interface>
  // <public>
  /**
    Deletes this thread removing also its reference in the document's collection.
  */
  @Override
  public boolean delete(
    )
  {
    // Shallow removal (references):
    // * reference in document
    getDocument().getArticles().remove(this);

    // Deep removal (indirect object).
    return super.delete();
  }

  /**
    Gets the beads associated to this thread.
  */
  public ArticleElements getElements(
    )
  {return ArticleElements.wrap(getBaseObject());}

  /**
    Gets common article metadata.
  */
  public Information getInformation(
    )
  {return Information.wrap(getBaseDataObject().get(PdfName.I, PdfDictionary.class));}

  /**
    @see #getInformation()
  */
  public void setInformation(
    Information value
    )
  {getBaseDataObject().put(PdfName.I, PdfObjectWrapper.getBaseObject(value));}
  // </public>
  // </interface>
  // </dynamic>
  // </class>
}
